# Docker Layer Caching Strategies

## Принцип работы кеширования

Docker кеширует каждую инструкцию как слой. Если слой изменяется — все последующие слои инвалидируются.

```
Слой 1: FROM node:22-alpine          ✓ Кешируется
Слой 2: COPY package.json ./         ✓ Кешируется (если файл не изменился)
Слой 3: RUN npm install              ✓ Кешируется (если слой 2 не изменился)
Слой 4: COPY . .                     ✗ Инвалидируется при любом изменении кода
Слой 5: RUN npm build                ✗ Пересобирается (зависит от слоя 4)
```

## Золотое правило

**Копируй файлы зависимостей ПЕРЕД кодом приложения**

### Плохо (инвалидирует npm install при каждом изменении)

```dockerfile
COPY . .
RUN npm install
RUN npm build
```

### Хорошо (npm install кешируется)

```dockerfile
COPY package.json package-lock.json ./
RUN npm install
COPY . .
RUN npm build
```

## BuildKit Cache Mounts

BuildKit позволяет монтировать кеш директории между сборками.

### Node.js npm/yarn cache

```dockerfile
# syntax=docker/dockerfile:1
FROM node:22-alpine

WORKDIR /app
COPY package.json package-lock.json ./

# Кешируем npm cache между сборками
RUN --mount=type=cache,target=/root/.npm \
    npm ci

COPY . .
RUN npm run build
```

### Ruby Bundler cache

```dockerfile
# syntax=docker/dockerfile:1
FROM ruby:3.3-alpine

WORKDIR /app
COPY Gemfile Gemfile.lock ./

# Кешируем gems между сборками
RUN --mount=type=cache,target=/root/.bundle \
    bundle config set --local deployment true && \
    bundle install --jobs 4

COPY . .
```

### Python pip cache

```dockerfile
# syntax=docker/dockerfile:1
FROM python:3.12-alpine

WORKDIR /app
COPY requirements.txt ./

# Кешируем pip cache
RUN --mount=type=cache,target=/root/.cache/pip \
    pip install -r requirements.txt

COPY . .
```

### Go modules cache

```dockerfile
# syntax=docker/dockerfile:1
FROM golang:1.22-alpine

WORKDIR /app
COPY go.mod go.sum ./

# Кешируем Go modules и build cache
RUN --mount=type=cache,target=/go/pkg/mod \
    --mount=type=cache,target=/root/.cache/go-build \
    go mod download

COPY . .
RUN --mount=type=cache,target=/root/.cache/go-build \
    go build -o /app/server ./cmd/server
```

### APT cache (Debian/Ubuntu)

```dockerfile
# syntax=docker/dockerfile:1
FROM debian:bookworm-slim

# Кешируем apt packages (sharing=locked для параллельных сборок)
RUN --mount=type=cache,target=/var/cache/apt,sharing=locked \
    --mount=type=cache,target=/var/lib/apt,sharing=locked \
    apt-get update && \
    apt-get install -y --no-install-recommends \
        curl \
        git

# Не удаляем /var/lib/apt/lists — это часть кеша
```

### APK cache (Alpine)

```dockerfile
# syntax=docker/dockerfile:1
FROM alpine:3.19

# Кешируем apk packages
RUN --mount=type=cache,target=/var/cache/apk \
    apk add --cache-dir=/var/cache/apk \
        curl \
        git \
        openssh-client
```

## COPY --link для улучшенного кеширования

`--link` создаёт независимый слой, который не инвалидируется при изменении предыдущих слоёв.

```dockerfile
# syntax=docker/dockerfile:1

FROM node:22-alpine AS builder
WORKDIR /app
COPY package.json package-lock.json ./
RUN npm ci
COPY . .
RUN npm run build

FROM nginx:alpine AS production
# --link позволяет кешировать этот слой независимо
COPY --link --from=builder /app/dist /usr/share/nginx/html
```

## Bind Mounts для зависимостей

Используй bind mounts вместо COPY для файлов, которые нужны только во время install:

```dockerfile
# syntax=docker/dockerfile:1
FROM node:22-alpine

WORKDIR /app

# Bind mount package files — не создаёт слой
RUN --mount=type=bind,source=package.json,target=package.json \
    --mount=type=bind,source=package-lock.json,target=package-lock.json \
    --mount=type=cache,target=/root/.npm \
    npm ci --only=production

COPY . .
```

## .dockerignore для минимизации контекста

Правильный `.dockerignore` ускоряет сборку и улучшает кеширование:

```dockerignore
# Git
.git
.gitignore

# Dependencies (устанавливаем внутри контейнера)
node_modules
vendor
__pycache__

# Build artifacts
dist
build
*.egg-info

# Development files
.env.local
.env.*.local
*.log

# IDE
.idea
.vscode
*.swp

# Docker
Dockerfile*
docker-compose*
.docker

# Documentation
README.md
docs/
*.md

# Tests (если не нужны в production)
test/
tests/
spec/
__tests__
coverage/
.nyc_output

# CI/CD
.github
.gitlab-ci.yml
.circleci
```

## Порядок инструкций

Упорядочивай инструкции от редко меняющихся к часто меняющимся:

```dockerfile
# 1. Базовый образ (меняется редко)
FROM node:22-alpine

# 2. System dependencies (меняются редко)
RUN apk add --no-cache curl

# 3. Working directory (никогда не меняется)
WORKDIR /app

# 4. Package files (меняются при изменении зависимостей)
COPY package.json package-lock.json ./

# 5. Install dependencies (кешируется если package.json не изменился)
RUN npm ci

# 6. Application code (меняется часто)
COPY . .

# 7. Build (выполняется при каждом изменении кода)
RUN npm run build

# 8. Runtime config (меняется редко)
EXPOSE 3000
CMD ["node", "dist/index.js"]
```

## Объединение RUN команд

Объединяй связанные команды для минимизации слоёв:

### Плохо (много слоёв)

```dockerfile
RUN apk update
RUN apk add curl
RUN apk add git
RUN rm -rf /var/cache/apk/*
```

### Хорошо (один слой)

```dockerfile
RUN apk add --no-cache \
    curl \
    git
```

## Multi-Stage для изоляции кеша

Используй отдельные stages для разных типов зависимостей:

```dockerfile
# syntax=docker/dockerfile:1

# Stage: Production dependencies
FROM node:22-alpine AS prod-deps
WORKDIR /app
COPY package.json package-lock.json ./
RUN --mount=type=cache,target=/root/.npm \
    npm ci --only=production

# Stage: Development dependencies (включает prod)
FROM node:22-alpine AS dev-deps
WORKDIR /app
COPY package.json package-lock.json ./
RUN --mount=type=cache,target=/root/.npm \
    npm ci

# Stage: Builder
FROM dev-deps AS builder
COPY . .
RUN npm run build

# Stage: Production
FROM node:22-alpine AS production
WORKDIR /app
COPY --from=prod-deps /app/node_modules ./node_modules
COPY --from=builder /app/dist ./dist
COPY package.json ./
CMD ["node", "dist/index.js"]
```

## Проверка эффективности кеширования

```bash
# Посмотреть историю слоёв
docker history myimage:latest

# Время сборки с кешем vs без
time docker build .
time docker build --no-cache .

# Анализ размера слоёв
docker image inspect myimage:latest --format '{{.Size}}'

# Детальный анализ с dive
dive myimage:latest
```

## CI/CD Cache Configuration

### GitHub Actions

```yaml
- name: Set up Docker Buildx
  uses: docker/setup-buildx-action@v3

- name: Build and push
  uses: docker/build-push-action@v5
  with:
    context: .
    cache-from: type=gha
    cache-to: type=gha,mode=max
```

### GitLab CI

```yaml
build:
  script:
    - docker build
        --cache-from $CI_REGISTRY_IMAGE:cache
        --tag $CI_REGISTRY_IMAGE:cache
        --tag $CI_REGISTRY_IMAGE:$CI_COMMIT_SHA
        .
    - docker push $CI_REGISTRY_IMAGE:cache
    - docker push $CI_REGISTRY_IMAGE:$CI_COMMIT_SHA
```
