# Docker Compose Best Practices

## Базовая структура compose.yaml

```yaml
# Версия не указывается в Compose V2+
name: myproject

services:
  app:
    build:
      context: ./app
      target: production
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=production
    depends_on:
      db:
        condition: service_healthy
    healthcheck:
      test: ["CMD", "wget", "--spider", "-q", "http://localhost:3000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s

  db:
    image: postgres:16-alpine
    volumes:
      - postgres_data:/var/lib/postgresql/data
    environment:
      POSTGRES_DB: ${DB_NAME}
      POSTGRES_USER: ${DB_USER}
      POSTGRES_PASSWORD: ${DB_PASSWORD}
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U ${DB_USER} -d ${DB_NAME}"]
      interval: 10s
      timeout: 5s
      retries: 5

volumes:
  postgres_data:

networks:
  default:
    name: myproject_network
```

## Multi-Stage Build в Compose

```yaml
services:
  # Development конфигурация
  app-dev:
    build:
      context: ./app
      target: development
    volumes:
      - ./app:/app
      - /app/node_modules  # Изолируем node_modules
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=development
    command: npm run dev

  # Production конфигурация
  app-prod:
    build:
      context: ./app
      target: production
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=production
```

## Shared Base Image Pattern

Один Dockerfile для нескольких сервисов:

```dockerfile
# Dockerfile.services
FROM alpine AS base
RUN apk add --no-cache openssl ca-certificates

FROM base AS service_a
COPY ./service_a /app
CMD ["/app/service_a"]

FROM base AS service_b
COPY ./service_b /app
CMD ["/app/service_b"]
```

```yaml
services:
  service_a:
    build:
      context: .
      dockerfile: Dockerfile.services
      target: service_a

  service_b:
    build:
      context: .
      dockerfile: Dockerfile.services
      target: service_b
```

## Profiles для опциональных сервисов

```yaml
services:
  app:
    build: ./app
    profiles: []  # Всегда запускается

  db:
    image: postgres:16-alpine
    profiles: []

  # Только для development
  adminer:
    image: adminer:latest
    profiles:
      - dev
    ports:
      - "8080:8080"

  # Только для тестирования
  test-runner:
    build:
      context: ./app
      target: test
    profiles:
      - test
    command: npm test

  # Мониторинг
  prometheus:
    image: prom/prometheus:latest
    profiles:
      - monitoring
```

Использование:
```bash
# Только основные сервисы
docker compose up

# С development tools
docker compose --profile dev up

# С тестами
docker compose --profile test run test-runner

# Несколько профилей
docker compose --profile dev --profile monitoring up
```

## Healthchecks

### Различные типы проверок

```yaml
services:
  # HTTP healthcheck
  web:
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:3000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s

  # TCP healthcheck
  redis:
    image: redis:7-alpine
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 10s
      timeout: 5s
      retries: 5

  # PostgreSQL
  postgres:
    image: postgres:16-alpine
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U $${POSTGRES_USER} -d $${POSTGRES_DB}"]
      interval: 10s
      timeout: 5s
      retries: 5

  # Shell command
  worker:
    healthcheck:
      test: ["CMD-SHELL", "pgrep -f 'sidekiq' || exit 1"]
      interval: 30s
      timeout: 10s
      retries: 3
```

## depends_on с условиями

```yaml
services:
  app:
    depends_on:
      db:
        condition: service_healthy
      redis:
        condition: service_healthy
      migrations:
        condition: service_completed_successfully

  migrations:
    build: ./app
    command: rails db:migrate
    depends_on:
      db:
        condition: service_healthy
```

## Environment Variables

### Из .env файла
```yaml
services:
  app:
    environment:
      - DATABASE_URL=postgres://${DB_USER}:${DB_PASSWORD}@db:5432/${DB_NAME}
      - REDIS_URL=redis://redis:6379/0
```

### Из env_file
```yaml
services:
  app:
    env_file:
      - .env
      - .env.local  # Переопределяет .env
```

### Дефолтные значения
```yaml
services:
  app:
    environment:
      - LOG_LEVEL=${LOG_LEVEL:-info}
      - PORT=${PORT:-3000}
```

## Volumes

### Named volumes (персистентные данные)
```yaml
volumes:
  postgres_data:
    driver: local
  redis_data:
    driver: local

services:
  db:
    volumes:
      - postgres_data:/var/lib/postgresql/data
```

### Bind mounts (development)
```yaml
services:
  app:
    volumes:
      - ./src:/app/src:ro          # Read-only source
      - ./config:/app/config:ro
      - /app/node_modules          # Anonymous volume (изоляция)
```

### tmpfs (временные данные)
```yaml
services:
  app:
    tmpfs:
      - /tmp
      - /app/tmp:size=100M
```

## Networks

```yaml
networks:
  frontend:
    driver: bridge
  backend:
    driver: bridge
    internal: true  # Без доступа к интернету

services:
  nginx:
    networks:
      - frontend

  app:
    networks:
      - frontend
      - backend

  db:
    networks:
      - backend  # Только внутренняя сеть
```

## Resource Limits

```yaml
services:
  app:
    deploy:
      resources:
        limits:
          cpus: '2'
          memory: 1G
        reservations:
          cpus: '0.5'
          memory: 256M
```

## Logging

```yaml
services:
  app:
    logging:
      driver: json-file
      options:
        max-size: "10m"
        max-file: "3"
```

## Restart Policies

```yaml
services:
  app:
    restart: unless-stopped  # Production

  worker:
    restart: on-failure      # Background jobs

  dev:
    restart: "no"            # Development
```

## Полный Production-Ready пример

```yaml
name: production-app

services:
  nginx:
    image: nginx:alpine
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./nginx/nginx.conf:/etc/nginx/nginx.conf:ro
      - ./nginx/ssl:/etc/nginx/ssl:ro
    depends_on:
      app:
        condition: service_healthy
    restart: unless-stopped
    logging:
      driver: json-file
      options:
        max-size: "10m"
        max-file: "3"

  app:
    build:
      context: ./app
      target: production
    environment:
      - DATABASE_URL=postgres://${DB_USER}:${DB_PASSWORD}@db:5432/${DB_NAME}
      - REDIS_URL=redis://redis:6379/0
      - SECRET_KEY_BASE=${SECRET_KEY_BASE}
    depends_on:
      db:
        condition: service_healthy
      redis:
        condition: service_healthy
    healthcheck:
      test: ["CMD", "wget", "--spider", "-q", "http://localhost:3000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s
    restart: unless-stopped
    deploy:
      resources:
        limits:
          cpus: '2'
          memory: 1G
    logging:
      driver: json-file
      options:
        max-size: "50m"
        max-file: "5"

  worker:
    build:
      context: ./app
      target: production
    command: bundle exec sidekiq
    environment:
      - DATABASE_URL=postgres://${DB_USER}:${DB_PASSWORD}@db:5432/${DB_NAME}
      - REDIS_URL=redis://redis:6379/0
    depends_on:
      db:
        condition: service_healthy
      redis:
        condition: service_healthy
    healthcheck:
      test: ["CMD-SHELL", "pgrep -f sidekiq || exit 1"]
      interval: 30s
      timeout: 10s
      retries: 3
    restart: unless-stopped
    deploy:
      resources:
        limits:
          cpus: '1'
          memory: 512M

  db:
    image: postgres:16-alpine
    volumes:
      - postgres_data:/var/lib/postgresql/data
    environment:
      POSTGRES_DB: ${DB_NAME}
      POSTGRES_USER: ${DB_USER}
      POSTGRES_PASSWORD: ${DB_PASSWORD}
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U ${DB_USER} -d ${DB_NAME}"]
      interval: 10s
      timeout: 5s
      retries: 5
    restart: unless-stopped

  redis:
    image: redis:7-alpine
    command: redis-server --appendonly yes
    volumes:
      - redis_data:/data
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 10s
      timeout: 5s
      retries: 5
    restart: unless-stopped

volumes:
  postgres_data:
  redis_data:

networks:
  default:
    name: production_network
```
