# Dockerfile Best Practices

## Multi-Stage Build Pattern

### Структура стадий

```dockerfile
# syntax=docker/dockerfile:1

# ============================================
# Stage 1: Dependencies (кешируемый слой)
# ============================================
FROM node:22-alpine AS deps
WORKDIR /app

# Копируем ТОЛЬКО файлы зависимостей (максимальный кеш)
COPY package.json package-lock.json ./
RUN npm ci --only=production

# ============================================
# Stage 2: Builder (сборка приложения)
# ============================================
FROM node:22-alpine AS builder
WORKDIR /app

COPY package.json package-lock.json ./
RUN npm ci

COPY . .
RUN npm run build

# ============================================
# Stage 3: Production (минимальный образ)
# ============================================
FROM node:22-alpine AS production
WORKDIR /app

# Security: non-root user
RUN addgroup -g 1001 -S appgroup && \
    adduser -u 1001 -S appuser -G appgroup

# Копируем только необходимое
COPY --from=deps /app/node_modules ./node_modules
COPY --from=builder /app/dist ./dist
COPY package.json ./

USER appuser
EXPOSE 3000

HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
  CMD wget --no-verbose --tries=1 --spider http://localhost:3000/health || exit 1

CMD ["node", "dist/index.js"]
```

## Node.js Patterns

### Development vs Production

```dockerfile
# syntax=docker/dockerfile:1
ARG NODE_VERSION=22

# Base stage
FROM node:${NODE_VERSION}-alpine AS base
WORKDIR /app
EXPOSE 3000

# Development stage
FROM base AS development
RUN --mount=type=bind,source=package.json,target=package.json \
    --mount=type=bind,source=package-lock.json,target=package-lock.json \
    --mount=type=cache,target=/root/.npm \
    npm ci --include=dev
USER node
COPY . .
CMD ["npm", "run", "dev"]

# Production stage
FROM base AS production
RUN --mount=type=bind,source=package.json,target=package.json \
    --mount=type=bind,source=package-lock.json,target=package-lock.json \
    --mount=type=cache,target=/root/.npm \
    npm ci --omit=dev
USER node
COPY . .
CMD ["node", "src/index.js"]
```

## Ruby on Rails Pattern

```dockerfile
# syntax=docker/dockerfile:1
ARG RUBY_VERSION=3.3

# ============================================
# Base stage
# ============================================
FROM ruby:${RUBY_VERSION}-alpine AS base
WORKDIR /app

# Runtime dependencies
RUN apk add --no-cache \
    postgresql-client \
    tzdata \
    libpq

# ============================================
# Build stage
# ============================================
FROM base AS builder

# Build dependencies
RUN apk add --no-cache \
    build-base \
    postgresql-dev \
    git

# Install gems with cache mount
COPY Gemfile Gemfile.lock ./
RUN --mount=type=cache,target=/root/.bundle \
    bundle config set --local deployment true && \
    bundle config set --local without 'development test' && \
    bundle install --jobs 4 --retry 3

COPY . .

# Precompile assets (if Rails with assets)
RUN SECRET_KEY_BASE=dummy bundle exec rails assets:precompile

# ============================================
# Production stage
# ============================================
FROM base AS production

# Security: non-root user
RUN addgroup -g 1001 -S rails && \
    adduser -u 1001 -S rails -G rails

COPY --from=builder /app /app
COPY --from=builder /usr/local/bundle /usr/local/bundle

USER rails
EXPOSE 3000

HEALTHCHECK --interval=30s --timeout=3s --start-period=10s --retries=3 \
  CMD wget --no-verbose --tries=1 --spider http://localhost:3000/health || exit 1

CMD ["bundle", "exec", "rails", "server", "-b", "0.0.0.0"]
```

## React/Vite Pattern (Static + Nginx)

```dockerfile
# syntax=docker/dockerfile:1

# ============================================
# Build stage
# ============================================
FROM node:22-alpine AS builder
WORKDIR /app

COPY package.json package-lock.json ./
RUN --mount=type=cache,target=/root/.npm \
    npm ci

COPY . .
RUN npm run build

# ============================================
# Production stage (Nginx)
# ============================================
FROM nginx:alpine AS production

# Custom nginx config
COPY nginx.conf /etc/nginx/nginx.conf

# Copy built assets
COPY --from=builder /app/dist /usr/share/nginx/html

# Security: non-root
RUN chown -R nginx:nginx /usr/share/nginx/html && \
    chmod -R 755 /usr/share/nginx/html

EXPOSE 80

HEALTHCHECK --interval=30s --timeout=3s --retries=3 \
  CMD wget --no-verbose --tries=1 --spider http://localhost/ || exit 1

CMD ["nginx", "-g", "daemon off;"]
```

## Go Pattern (Scratch Final Image)

```dockerfile
# syntax=docker/dockerfile:1

# ============================================
# Build stage
# ============================================
FROM golang:1.22-alpine AS builder
WORKDIR /app

# Cache dependencies
COPY go.mod go.sum ./
RUN --mount=type=cache,target=/go/pkg/mod \
    go mod download

COPY . .

# Build static binary
RUN --mount=type=cache,target=/root/.cache/go-build \
    CGO_ENABLED=0 GOOS=linux go build -ldflags="-w -s" -o /app/server ./cmd/server

# ============================================
# Production stage (scratch = минимальный образ)
# ============================================
FROM scratch AS production

# Copy CA certificates for HTTPS
COPY --from=builder /etc/ssl/certs/ca-certificates.crt /etc/ssl/certs/

# Copy binary
COPY --from=builder /app/server /server

EXPOSE 8080

ENTRYPOINT ["/server"]
```

## Alpine Специфика

### Установка пакетов

```dockerfile
# Правильно: --no-cache для минимизации размера
RUN apk add --no-cache \
    curl \
    git \
    openssh-client

# Для build dependencies: virtual package
RUN apk add --no-cache --virtual .build-deps \
    build-base \
    gcc \
    && npm install \
    && apk del .build-deps
```

### Частые проблемы Alpine

```dockerfile
# DNS resolution в Alpine
RUN echo "hosts: files dns" > /etc/nsswitch.conf

# Timezone
RUN apk add --no-cache tzdata
ENV TZ=UTC

# Locale (Alpine не поддерживает glibc locales)
ENV LANG=C.UTF-8
ENV LC_ALL=C.UTF-8
```

## ARG и ENV Best Practices

```dockerfile
# ARG — доступен только во время build
ARG NODE_VERSION=22
ARG APP_ENV=production

# ENV — доступен в runtime
ENV NODE_ENV=${APP_ENV}
ENV PORT=3000

# Комбинация ARG → ENV
ARG VERSION
ENV APP_VERSION=${VERSION:-unknown}
```

## COPY vs ADD

```dockerfile
# COPY — предпочтительно для файлов
COPY package.json ./
COPY src/ ./src/

# ADD — только для tar архивов (авто-распаковка)
ADD app.tar.gz /app/

# COPY --link — улучшенное кеширование (BuildKit)
COPY --link package.json ./
```

## Labels для метаданных

```dockerfile
LABEL org.opencontainers.image.source="https://github.com/org/repo"
LABEL org.opencontainers.image.description="Application description"
LABEL org.opencontainers.image.version="1.0.0"
LABEL org.opencontainers.image.authors="team@example.com"
```
