# Docker Security Best Practices

## Non-Root User

**Никогда не запускай приложение от root в production.**

### Создание пользователя в Alpine

```dockerfile
# Создаём группу и пользователя
RUN addgroup -g 1001 -S appgroup && \
    adduser -u 1001 -S appuser -G appgroup

# Устанавливаем владельца файлов
COPY --chown=appuser:appgroup . /app

# Переключаемся на пользователя
USER appuser
```

### Создание пользователя в Debian/Ubuntu

```dockerfile
# Создаём системного пользователя без shell и home
RUN adduser \
    --disabled-password \
    --gecos "" \
    --home "/nonexistent" \
    --shell "/sbin/nologin" \
    --no-create-home \
    --uid 1001 \
    appuser

USER appuser
```

### Node.js с встроенным пользователем

```dockerfile
FROM node:22-alpine

WORKDIR /app
COPY --chown=node:node . .

# Node образы уже имеют пользователя 'node'
USER node
```

## Фиксированные версии образов

### Плохо
```dockerfile
FROM node:latest        # Непредсказуемо
FROM node:22           # Может измениться minor версия
```

### Хорошо
```dockerfile
FROM node:22.11.0-alpine3.19    # Полная версия
FROM node:22-alpine             # Допустимо для dev
```

### Использование digest (максимальная воспроизводимость)
```dockerfile
FROM node:22-alpine@sha256:abc123...
```

## Минимизация Attack Surface

### Удаление лишних пакетов

```dockerfile
# Alpine: используй --no-cache
RUN apk add --no-cache --virtual .build-deps \
    build-base \
    gcc \
    && npm install \
    && apk del .build-deps
```

### Удаление shell в production (экстремальный вариант)

```dockerfile
# Для Go/статических бинарников
FROM scratch
COPY --from=builder /app/server /server
# В scratch нет shell, нет wget, нет ничего
```

### Distroless образы (Google)

```dockerfile
FROM gcr.io/distroless/nodejs22-debian12
COPY --from=builder /app/dist /app
CMD ["app/index.js"]
```

## Secrets Management

### Никогда не храни секреты в образе

```dockerfile
# ПЛОХО — секреты останутся в слое
COPY .env /app/.env
ENV API_KEY=secret123

# ПЛОХО — даже после удаления видно в истории
COPY .env /app/.env
RUN cat .env && rm .env
```

### BuildKit secrets (для build-time)

```dockerfile
# syntax=docker/dockerfile:1

RUN --mount=type=secret,id=npmrc,target=/root/.npmrc \
    npm install
```

```bash
docker build --secret id=npmrc,src=.npmrc .
```

### Runtime secrets через environment/volumes

```yaml
# docker-compose.yml
services:
  app:
    environment:
      - DATABASE_URL  # Из .env или системы
    secrets:
      - db_password

secrets:
  db_password:
    file: ./secrets/db_password.txt
```

## Read-Only Filesystem

```yaml
services:
  app:
    read_only: true
    tmpfs:
      - /tmp
      - /app/tmp
    volumes:
      - logs:/app/logs  # Только для записи логов
```

```dockerfile
# В Dockerfile — минимизируй записываемые директории
RUN mkdir -p /app/tmp /app/logs && \
    chown -R appuser:appgroup /app/tmp /app/logs
```

## Capabilities

```yaml
services:
  app:
    cap_drop:
      - ALL
    cap_add:
      - NET_BIND_SERVICE  # Только если нужен порт < 1024
```

## Security Options

```yaml
services:
  app:
    security_opt:
      - no-new-privileges:true
```

## Healthchecks (для обнаружения компрометации)

```dockerfile
HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
  CMD wget --no-verbose --tries=1 --spider http://localhost:3000/health || exit 1
```

## Сканирование уязвимостей

### Docker Scout

```bash
docker scout cves myimage:latest
docker scout recommendations myimage:latest
```

### Trivy

```bash
trivy image myimage:latest
trivy image --severity HIGH,CRITICAL myimage:latest
```

### CI/CD Integration

```yaml
# GitHub Actions
- name: Scan for vulnerabilities
  uses: aquasecurity/trivy-action@master
  with:
    image-ref: myimage:latest
    severity: 'CRITICAL,HIGH'
    exit-code: '1'
```

## Network Security

```yaml
networks:
  frontend:
    driver: bridge
  backend:
    driver: bridge
    internal: true  # Без доступа к интернету

services:
  app:
    networks:
      - frontend
      - backend

  db:
    networks:
      - backend  # Изолирована от интернета
```

## Полный Security-First Dockerfile

```dockerfile
# syntax=docker/dockerfile:1
FROM node:22-alpine AS builder

WORKDIR /app

# Установка зависимостей
COPY package.json package-lock.json ./
RUN --mount=type=cache,target=/root/.npm \
    npm ci

COPY . .
RUN npm run build && \
    npm prune --production

# Production stage
FROM node:22-alpine AS production

# Security: обновляем базовые пакеты
RUN apk upgrade --no-cache

# Security: non-root user
RUN addgroup -g 1001 -S nodejs && \
    adduser -u 1001 -S nodejs -G nodejs

WORKDIR /app

# Копируем с правильным владельцем
COPY --from=builder --chown=nodejs:nodejs /app/dist ./dist
COPY --from=builder --chown=nodejs:nodejs /app/node_modules ./node_modules
COPY --from=builder --chown=nodejs:nodejs /app/package.json ./

# Security: переключаемся на non-root
USER nodejs

# Security: фиксируем порт
EXPOSE 3000

# Security: healthcheck
HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
  CMD wget --no-verbose --tries=1 --spider http://localhost:3000/health || exit 1

CMD ["node", "dist/index.js"]
```

## Security Checklist

- [ ] Non-root пользователь (USER instruction)
- [ ] Фиксированные версии базовых образов
- [ ] Нет секретов в образе (ARG/ENV/COPY)
- [ ] Минимум установленных пакетов
- [ ] Multi-stage build (без build tools в production)
- [ ] HEALTHCHECK определён
- [ ] Сканирование уязвимостей в CI/CD
- [ ] Read-only filesystem где возможно
- [ ] Сетевая изоляция (internal networks)
- [ ] Capabilities dropped
- [ ] no-new-privileges:true
