# syntax=docker/dockerfile:1
#
# Go Multi-Stage Dockerfile
# Produces minimal scratch-based image
#

ARG GO_VERSION=1.22

# ============================================
# Stage: Builder
# ============================================
FROM golang:${GO_VERSION}-alpine AS builder
WORKDIR /app

# Install git for private modules (if needed)
RUN apk add --no-cache git ca-certificates

# Download dependencies (cached layer)
COPY go.mod go.sum ./
RUN --mount=type=cache,target=/go/pkg/mod \
    go mod download

# Copy source code
COPY . .

# Build static binary
RUN --mount=type=cache,target=/root/.cache/go-build \
    CGO_ENABLED=0 GOOS=linux GOARCH=amd64 \
    go build -ldflags="-w -s" -o /app/server ./cmd/server

# ============================================
# Stage: Development
# ============================================
FROM golang:${GO_VERSION}-alpine AS development
WORKDIR /app

RUN apk add --no-cache git

COPY go.mod go.sum ./
RUN go mod download

COPY . .

EXPOSE 8080
CMD ["go", "run", "./cmd/server"]

# ============================================
# Stage: Production (scratch)
# ============================================
FROM scratch AS production

# Copy CA certificates for HTTPS
COPY --from=builder /etc/ssl/certs/ca-certificates.crt /etc/ssl/certs/

# Copy binary
COPY --from=builder /app/server /server

# Non-root user (numeric for scratch)
USER 1001:1001

EXPOSE 8080

# Note: HEALTHCHECK not available in scratch
# Use orchestrator health checks (Kubernetes, Docker Compose)

ENTRYPOINT ["/server"]
