# Архитектура проекта

Feature-Sliced Design адаптированный для небольших/средних проектов.

## Структура папок

```
src/
├── app/                        # Application layer
│   ├── App.tsx                 # Root component
│   ├── providers/              # Context providers
│   │   ├── AuthProvider.tsx
│   │   ├── ThemeProvider.tsx
│   │   └── index.tsx           # Combined providers
│   ├── router/                 # Routing configuration
│   │   ├── routes.tsx
│   │   └── ProtectedRoute.tsx
│   └── styles/                 # Global styles
│       └── index.css
│
├── components/                 # Shared components
│   ├── ui/                     # Base UI primitives
│   │   ├── Button/
│   │   │   ├── Button.tsx
│   │   │   ├── Button.test.tsx
│   │   │   └── index.ts
│   │   ├── Input/
│   │   ├── Modal/
│   │   └── index.ts            # Barrel export
│   ├── layout/                 # Layout components
│   │   ├── Header.tsx
│   │   ├── Sidebar.tsx
│   │   └── PageLayout.tsx
│   └── common/                 # Shared business components
│       ├── UserAvatar.tsx
│       └── LoadingSpinner.tsx
│
├── features/                   # Feature modules
│   ├── auth/
│   │   ├── components/
│   │   │   ├── LoginForm.tsx
│   │   │   └── RegisterForm.tsx
│   │   ├── hooks/
│   │   │   ├── useAuth.ts
│   │   │   └── useLogin.ts
│   │   ├── api/
│   │   │   └── authApi.ts
│   │   ├── types.ts
│   │   └── index.ts
│   │
│   └── homework/
│       ├── components/
│       │   ├── HomeworkCard.tsx
│       │   ├── HomeworkList.tsx
│       │   └── SubmitHomework.tsx
│       ├── hooks/
│       │   └── useHomework.ts
│       ├── api/
│       │   └── homeworkApi.ts
│       └── index.ts
│
├── hooks/                      # Shared hooks
│   ├── useLocalStorage.ts
│   ├── useDebounce.ts
│   └── index.ts
│
├── lib/                        # Utilities
│   ├── api.ts                  # API client (axios)
│   ├── utils.ts                # Helper functions
│   └── constants.ts
│
├── pages/                      # Page components (routes)
│   ├── HomePage.tsx
│   ├── LoginPage.tsx
│   └── DashboardPage.tsx
│
├── services/                   # External services
│   ├── telegram.ts             # Telegram Mini App SDK
│   └── analytics.ts
│
├── stores/                     # Global state (если нужен)
│   └── userStore.ts
│
├── types/                      # Global types
│   ├── api.ts
│   └── common.ts
│
└── main.tsx                    # Entry point
```

## Правила организации

### 1. Feature Module

Каждая фича — изолированный модуль:

```typescript
// features/auth/index.ts — публичный API фичи
export { LoginForm } from './components/LoginForm'
export { useAuth } from './hooks/useAuth'
export type { User, AuthState } from './types'

// Не экспортируй внутренние компоненты!
```

### 2. Импорты

```typescript
// ✅ Хорошо — через публичный API
import { LoginForm, useAuth } from '@/features/auth'

// ❌ Плохо — прямой импорт внутренних файлов
import { LoginForm } from '@/features/auth/components/LoginForm'
```

### 3. Barrel Exports

```typescript
// components/ui/index.ts
export { Button } from './Button'
export { Input } from './Input'
export { Modal } from './Modal'

// Использование
import { Button, Input } from '@/components/ui'
```

### 4. Path Aliases

```typescript
// vite.config.ts
export default defineConfig({
  resolve: {
    alias: {
      '@': path.resolve(__dirname, './src'),
    },
  },
})

// tsconfig.json
{
  "compilerOptions": {
    "baseUrl": ".",
    "paths": {
      "@/*": ["src/*"]
    }
  }
}
```

## Зависимости между слоями

```
pages → features → components → ui
         ↓
       hooks
         ↓
     lib/services
```

- **pages** могут импортировать из features, components
- **features** могут импортировать из components, hooks, lib
- **components** могут импортировать из ui, hooks, lib
- **ui** — только lib, никаких бизнес-зависимостей

## Именование файлов

```
PascalCase.tsx    — компоненты
camelCase.ts      — хуки, утилиты, сервисы
kebab-case/       — папки (опционально)
```

## Типичный Feature Module

```
features/homework/
├── components/
│   ├── HomeworkCard.tsx        # Карточка задания
│   ├── HomeworkCard.test.tsx   # Тест компонента
│   ├── HomeworkList.tsx        # Список заданий
│   └── SubmitForm.tsx          # Форма отправки
├── hooks/
│   ├── useHomework.ts          # Получение списка
│   ├── useSubmitHomework.ts    # Отправка задания
│   └── useHomeworkStatus.ts    # Статус проверки
├── api/
│   └── homeworkApi.ts          # API endpoints
├── types.ts                    # Типы фичи
├── constants.ts                # Константы фичи
└── index.ts                    # Public API
```

## Entry Point

```tsx
// main.tsx
import React from 'react'
import ReactDOM from 'react-dom/client'
import { App } from './app/App'
import './app/styles/index.css'

ReactDOM.createRoot(document.getElementById('root')!).render(
  <React.StrictMode>
    <App />
  </React.StrictMode>
)
```

```tsx
// app/App.tsx
import { Providers } from './providers'
import { Router } from './router'

export function App() {
  return (
    <Providers>
      <Router />
    </Providers>
  )
}
```
