# Компоненты React

Правила написания чистых, переиспользуемых компонентов.

## Структура компонента

```tsx
// components/ui/Button/Button.tsx
import { forwardRef, type ComponentPropsWithoutRef } from 'react'
import { cn } from '@/lib/utils'

// 1. Types
interface ButtonProps extends ComponentPropsWithoutRef<'button'> {
  variant?: 'primary' | 'secondary' | 'danger'
  size?: 'sm' | 'md' | 'lg'
  isLoading?: boolean
}

// 2. Component
export const Button = forwardRef<HTMLButtonElement, ButtonProps>(
  ({ className, variant = 'primary', size = 'md', isLoading, children, disabled, ...props }, ref) => {
    return (
      <button
        ref={ref}
        className={cn(
          'inline-flex items-center justify-center rounded-md font-medium transition-colors',
          'focus-visible:outline-none focus-visible:ring-2',
          'disabled:pointer-events-none disabled:opacity-50',
          variants[variant],
          sizes[size],
          className
        )}
        disabled={disabled || isLoading}
        {...props}
      >
        {isLoading && <Spinner className="mr-2 h-4 w-4" />}
        {children}
      </button>
    )
  }
)

Button.displayName = 'Button'

// 3. Variants (вынесены для читаемости)
const variants = {
  primary: 'bg-blue-600 text-white hover:bg-blue-700',
  secondary: 'bg-gray-200 text-gray-900 hover:bg-gray-300',
  danger: 'bg-red-600 text-white hover:bg-red-700',
}

const sizes = {
  sm: 'h-8 px-3 text-sm',
  md: 'h-10 px-4 text-base',
  lg: 'h-12 px-6 text-lg',
}
```

## Правила

### 1. Single Responsibility

```tsx
// ❌ Плохо — компонент делает слишком много
function UserCard({ userId }) {
  const [user, setUser] = useState(null)
  const [posts, setPosts] = useState([])

  useEffect(() => { fetchUser(userId).then(setUser) }, [userId])
  useEffect(() => { fetchPosts(userId).then(setPosts) }, [userId])

  return (
    <div>
      <img src={user?.avatar} />
      <h2>{user?.name}</h2>
      {posts.map(post => <PostCard key={post.id} post={post} />)}
    </div>
  )
}

// ✅ Хорошо — разделение ответственности
function UserCard({ user }: { user: User }) {
  return (
    <div className="flex items-center gap-3">
      <UserAvatar src={user.avatar} alt={user.name} />
      <UserInfo name={user.name} email={user.email} />
    </div>
  )
}

function UserProfile({ userId }: { userId: string }) {
  const { user, isLoading } = useUser(userId)

  if (isLoading) return <Skeleton />
  if (!user) return <NotFound />

  return <UserCard user={user} />
}
```

### 2. Props Interface

```tsx
// ✅ Extend native props
interface InputProps extends ComponentPropsWithoutRef<'input'> {
  label?: string
  error?: string
}

// ✅ Дискриминантные union типы
type ButtonProps =
  | { variant: 'link'; href: string }
  | { variant: 'button'; onClick: () => void }

// ✅ Generics для типизированных компонентов
interface SelectProps<T> {
  options: T[]
  value: T
  onChange: (value: T) => void
  getLabel: (item: T) => string
  getValue: (item: T) => string
}
```

### 3. Composition

```tsx
// ✅ Compound Components
function Card({ children, className }: PropsWithChildren<{ className?: string }>) {
  return <div className={cn('rounded-lg border bg-white p-4', className)}>{children}</div>
}

Card.Header = function CardHeader({ children }: PropsWithChildren) {
  return <div className="mb-4 border-b pb-4">{children}</div>
}

Card.Body = function CardBody({ children }: PropsWithChildren) {
  return <div className="space-y-2">{children}</div>
}

Card.Footer = function CardFooter({ children }: PropsWithChildren) {
  return <div className="mt-4 flex justify-end gap-2">{children}</div>
}

// Использование
<Card>
  <Card.Header>
    <h2>Title</h2>
  </Card.Header>
  <Card.Body>
    <p>Content</p>
  </Card.Body>
  <Card.Footer>
    <Button>Save</Button>
  </Card.Footer>
</Card>
```

### 4. Render Props / Children as Function

```tsx
// Для сложной логики переиспользования
interface ToggleRenderProps {
  isOn: boolean
  toggle: () => void
}

function Toggle({ children }: { children: (props: ToggleRenderProps) => ReactNode }) {
  const [isOn, setIsOn] = useState(false)
  return children({ isOn, toggle: () => setIsOn(!isOn) })
}

// Использование
<Toggle>
  {({ isOn, toggle }) => (
    <button onClick={toggle}>
      {isOn ? 'ON' : 'OFF'}
    </button>
  )}
</Toggle>
```

### 5. Controlled vs Uncontrolled

```tsx
// Поддержка обоих режимов
interface InputProps {
  value?: string              // Controlled
  defaultValue?: string       // Uncontrolled
  onChange?: (value: string) => void
}

function Input({ value, defaultValue, onChange, ...props }: InputProps) {
  const [internalValue, setInternalValue] = useState(defaultValue ?? '')

  const isControlled = value !== undefined
  const currentValue = isControlled ? value : internalValue

  const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
    const newValue = e.target.value
    if (!isControlled) {
      setInternalValue(newValue)
    }
    onChange?.(newValue)
  }

  return <input value={currentValue} onChange={handleChange} {...props} />
}
```

## Паттерны

### Container / Presentational

```tsx
// Container — логика
function UserListContainer() {
  const { users, isLoading, error } = useUsers()
  const { deleteUser } = useDeleteUser()

  if (isLoading) return <Skeleton />
  if (error) return <ErrorMessage error={error} />

  return <UserList users={users} onDelete={deleteUser} />
}

// Presentational — UI
function UserList({ users, onDelete }: UserListProps) {
  return (
    <ul>
      {users.map(user => (
        <UserItem key={user.id} user={user} onDelete={() => onDelete(user.id)} />
      ))}
    </ul>
  )
}
```

### HOC (редко, предпочитай хуки)

```tsx
function withAuth<P extends object>(Component: ComponentType<P>) {
  return function AuthenticatedComponent(props: P) {
    const { user, isLoading } = useAuth()

    if (isLoading) return <Spinner />
    if (!user) return <Navigate to="/login" />

    return <Component {...props} />
  }
}
```

## TailwindCSS Utility

```typescript
// lib/utils.ts
import { clsx, type ClassValue } from 'clsx'
import { twMerge } from 'tailwind-merge'

export function cn(...inputs: ClassValue[]) {
  return twMerge(clsx(inputs))
}

// Использование
<div className={cn(
  'base-classes',
  isActive && 'active-classes',
  className  // props override
)} />
```

## Файловая структура компонента

```
Button/
├── Button.tsx          # Компонент
├── Button.test.tsx     # Unit тест
├── Button.stories.tsx  # Storybook (опционально)
└── index.ts            # Export
```

```typescript
// Button/index.ts
export { Button } from './Button'
export type { ButtonProps } from './Button'
```
