# E2E тестирование

Playwright для end-to-end тестирования.

## Настройка

```bash
npm init playwright@latest
```

```typescript
// playwright.config.ts
import { defineConfig, devices } from '@playwright/test'

export default defineConfig({
  testDir: './e2e',
  fullyParallel: true,
  forbidOnly: !!process.env.CI,
  retries: process.env.CI ? 2 : 0,
  workers: process.env.CI ? 1 : undefined,
  reporter: 'html',
  use: {
    baseURL: 'http://localhost:5173',
    trace: 'on-first-retry',
    screenshot: 'only-on-failure',
  },
  projects: [
    {
      name: 'chromium',
      use: { ...devices['Desktop Chrome'] },
    },
    {
      name: 'firefox',
      use: { ...devices['Desktop Firefox'] },
    },
    {
      name: 'mobile',
      use: { ...devices['iPhone 13'] },
    },
  ],
  webServer: {
    command: 'npm run dev',
    url: 'http://localhost:5173',
    reuseExistingServer: !process.env.CI,
  },
})
```

## Базовый тест

```typescript
// e2e/home.spec.ts
import { test, expect } from '@playwright/test'

test.describe('Home Page', () => {
  test('should display welcome message', async ({ page }) => {
    await page.goto('/')

    await expect(page.getByRole('heading', { name: /welcome/i })).toBeVisible()
  })

  test('should navigate to login', async ({ page }) => {
    await page.goto('/')

    await page.getByRole('link', { name: /sign in/i }).click()

    await expect(page).toHaveURL('/login')
  })
})
```

## Page Object Model

```typescript
// e2e/pages/LoginPage.ts
import { type Page, type Locator, expect } from '@playwright/test'

export class LoginPage {
  readonly page: Page
  readonly emailInput: Locator
  readonly passwordInput: Locator
  readonly submitButton: Locator
  readonly errorMessage: Locator

  constructor(page: Page) {
    this.page = page
    this.emailInput = page.getByLabel(/email/i)
    this.passwordInput = page.getByLabel(/password/i)
    this.submitButton = page.getByRole('button', { name: /sign in/i })
    this.errorMessage = page.getByRole('alert')
  }

  async goto() {
    await this.page.goto('/login')
  }

  async login(email: string, password: string) {
    await this.emailInput.fill(email)
    await this.passwordInput.fill(password)
    await this.submitButton.click()
  }

  async expectError(message: string | RegExp) {
    await expect(this.errorMessage).toContainText(message)
  }

  async expectLoggedIn() {
    await expect(this.page).toHaveURL('/dashboard')
  }
}
```

```typescript
// e2e/auth.spec.ts
import { test, expect } from '@playwright/test'
import { LoginPage } from './pages/LoginPage'

test.describe('Authentication', () => {
  let loginPage: LoginPage

  test.beforeEach(async ({ page }) => {
    loginPage = new LoginPage(page)
    await loginPage.goto()
  })

  test('should login with valid credentials', async () => {
    await loginPage.login('user@example.com', 'password123')
    await loginPage.expectLoggedIn()
  })

  test('should show error for invalid credentials', async () => {
    await loginPage.login('user@example.com', 'wrongpassword')
    await loginPage.expectError(/invalid credentials/i)
  })

  test('should validate required fields', async () => {
    await loginPage.submitButton.click()

    await expect(loginPage.emailInput).toHaveAttribute('aria-invalid', 'true')
  })
})
```

## Fixtures

```typescript
// e2e/fixtures.ts
import { test as base } from '@playwright/test'
import { LoginPage } from './pages/LoginPage'
import { DashboardPage } from './pages/DashboardPage'

type Fixtures = {
  loginPage: LoginPage
  dashboardPage: DashboardPage
  authenticatedPage: DashboardPage
}

export const test = base.extend<Fixtures>({
  loginPage: async ({ page }, use) => {
    const loginPage = new LoginPage(page)
    await use(loginPage)
  },

  dashboardPage: async ({ page }, use) => {
    const dashboardPage = new DashboardPage(page)
    await use(dashboardPage)
  },

  // Фикстура с предварительной аутентификацией
  authenticatedPage: async ({ page }, use) => {
    // Login programmatically
    await page.goto('/login')
    await page.getByLabel(/email/i).fill('test@example.com')
    await page.getByLabel(/password/i).fill('password123')
    await page.getByRole('button', { name: /sign in/i }).click()
    await page.waitForURL('/dashboard')

    const dashboardPage = new DashboardPage(page)
    await use(dashboardPage)
  },
})

export { expect } from '@playwright/test'
```

```typescript
// e2e/dashboard.spec.ts
import { test, expect } from './fixtures'

test.describe('Dashboard', () => {
  test('should show user data after login', async ({ authenticatedPage }) => {
    await expect(authenticatedPage.welcomeMessage).toBeVisible()
    await expect(authenticatedPage.userStats).toHaveCount(3)
  })
})
```

## Локаторы (Best Practices)

```typescript
// ✅ Хорошо — по роли (accessibility)
page.getByRole('button', { name: /submit/i })
page.getByRole('textbox', { name: /email/i })
page.getByRole('link', { name: /home/i })
page.getByRole('heading', { level: 1 })

// ✅ Хорошо — по label
page.getByLabel(/password/i)
page.getByPlaceholder(/search/i)

// ✅ Хорошо — по тексту
page.getByText(/welcome/i)
page.getByText('Submit', { exact: true })

// ✅ Фильтрация и цепочка
page.getByRole('listitem').filter({ hasText: 'Product' })
page.getByRole('listitem').filter({ hasText: 'Product' }).getByRole('button')

// ❌ Плохо — CSS селекторы (хрупкие)
page.locator('.btn-primary')
page.locator('#submit-btn')

// ⚠️ Последний resort — test-id
page.getByTestId('custom-element')
```

## Web-First Assertions

```typescript
// ✅ Используй web-first assertions (авто-ретрай)
await expect(page.getByText('Success')).toBeVisible()
await expect(page.getByRole('button')).toBeEnabled()
await expect(page).toHaveURL('/dashboard')
await expect(page).toHaveTitle(/Dashboard/)

// ❌ НЕ используй manual assertions
const isVisible = await page.getByText('Success').isVisible()
expect(isVisible).toBe(true) // Не ждёт!
```

## API Mocking

```typescript
// Мокирование API ответов
test('should display mocked data', async ({ page }) => {
  // Mock before navigation
  await page.route('**/api/users', async (route) => {
    await route.fulfill({
      status: 200,
      contentType: 'application/json',
      body: JSON.stringify([
        { id: 1, name: 'Mocked User' },
      ]),
    })
  })

  await page.goto('/users')

  await expect(page.getByText('Mocked User')).toBeVisible()
})

// Mock error response
test('should handle API error', async ({ page }) => {
  await page.route('**/api/users', async (route) => {
    await route.fulfill({
      status: 500,
      body: JSON.stringify({ error: 'Server Error' }),
    })
  })

  await page.goto('/users')

  await expect(page.getByText(/error loading/i)).toBeVisible()
})
```

## Authentication State

```typescript
// e2e/auth.setup.ts
import { test as setup, expect } from '@playwright/test'
import path from 'path'

const authFile = path.join(__dirname, '.auth/user.json')

setup('authenticate', async ({ page }) => {
  await page.goto('/login')
  await page.getByLabel(/email/i).fill('test@example.com')
  await page.getByLabel(/password/i).fill('password123')
  await page.getByRole('button', { name: /sign in/i }).click()
  await page.waitForURL('/dashboard')

  // Save auth state
  await page.context().storageState({ path: authFile })
})
```

```typescript
// playwright.config.ts
export default defineConfig({
  projects: [
    // Setup project
    { name: 'setup', testMatch: /.*\.setup\.ts/ },

    // Tests that require auth
    {
      name: 'chromium',
      use: {
        ...devices['Desktop Chrome'],
        storageState: 'e2e/.auth/user.json',
      },
      dependencies: ['setup'],
    },
  ],
})
```

## Visual Testing

```typescript
test('should match screenshot', async ({ page }) => {
  await page.goto('/dashboard')

  // Full page screenshot
  await expect(page).toHaveScreenshot('dashboard.png')

  // Element screenshot
  await expect(page.getByTestId('chart')).toHaveScreenshot('chart.png')
})

// С threshold для динамического контента
await expect(page).toHaveScreenshot('dashboard.png', {
  maxDiffPixels: 100,
})
```

## Структура E2E тестов

```
e2e/
├── fixtures.ts               # Custom fixtures
├── pages/                    # Page Objects
│   ├── LoginPage.ts
│   ├── DashboardPage.ts
│   └── index.ts
├── auth.setup.ts             # Auth setup
├── .auth/                    # Auth state (gitignored)
│   └── user.json
├── auth.spec.ts              # Auth tests
├── dashboard.spec.ts         # Dashboard tests
└── homework.spec.ts          # Feature tests
```

## Команды

```bash
# Запуск всех тестов
npx playwright test

# Запуск в UI режиме
npx playwright test --ui

# Запуск конкретного файла
npx playwright test auth.spec.ts

# Headed режим (видимый браузер)
npx playwright test --headed

# Debug режим
npx playwright test --debug

# Генерация тестов
npx playwright codegen localhost:5173

# Просмотр отчёта
npx playwright show-report
```

## Чеклист E2E тестов

- [ ] Page Objects для страниц
- [ ] Fixtures для переиспользуемых setup
- [ ] Роль-based локаторы (accessibility)
- [ ] Web-first assertions
- [ ] API mocking для изоляции
- [ ] Auth state для быстрых тестов
- [ ] Visual regression (опционально)
