# Кастомные хуки

Правила создания переиспользуемых хуков.

## Принципы

1. **Один хук — одна задача**
2. **Возвращай объект** для расширяемости
3. **Оборачивай функции в useCallback**
4. **Используй useMemo для вычислений**

## Базовые паттерны

### Data Fetching Hook

```typescript
// hooks/useApi.ts
import { useState, useEffect, useCallback } from 'react'

interface UseApiResult<T> {
  data: T | null
  isLoading: boolean
  error: Error | null
  refetch: () => Promise<void>
}

export function useApi<T>(
  fetcher: () => Promise<T>,
  deps: unknown[] = []
): UseApiResult<T> {
  const [data, setData] = useState<T | null>(null)
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<Error | null>(null)

  const fetch = useCallback(async () => {
    setIsLoading(true)
    setError(null)
    try {
      const result = await fetcher()
      setData(result)
    } catch (e) {
      setError(e instanceof Error ? e : new Error('Unknown error'))
    } finally {
      setIsLoading(false)
    }
  }, deps)

  useEffect(() => {
    fetch()
  }, [fetch])

  return { data, isLoading, error, refetch: fetch }
}

// Использование
function UserProfile({ userId }: { userId: string }) {
  const { data: user, isLoading, error, refetch } = useApi(
    () => api.getUser(userId),
    [userId]
  )

  if (isLoading) return <Spinner />
  if (error) return <Error error={error} onRetry={refetch} />

  return <UserCard user={user!} />
}
```

### Mutation Hook

```typescript
// hooks/useMutation.ts
interface UseMutationResult<TData, TVariables> {
  mutate: (variables: TVariables) => Promise<TData>
  data: TData | null
  isLoading: boolean
  error: Error | null
  reset: () => void
}

export function useMutation<TData, TVariables>(
  mutationFn: (variables: TVariables) => Promise<TData>
): UseMutationResult<TData, TVariables> {
  const [data, setData] = useState<TData | null>(null)
  const [isLoading, setIsLoading] = useState(false)
  const [error, setError] = useState<Error | null>(null)

  const mutate = useCallback(async (variables: TVariables) => {
    setIsLoading(true)
    setError(null)
    try {
      const result = await mutationFn(variables)
      setData(result)
      return result
    } catch (e) {
      const error = e instanceof Error ? e : new Error('Unknown error')
      setError(error)
      throw error
    } finally {
      setIsLoading(false)
    }
  }, [mutationFn])

  const reset = useCallback(() => {
    setData(null)
    setError(null)
    setIsLoading(false)
  }, [])

  return { mutate, data, isLoading, error, reset }
}

// Использование
function DeleteButton({ userId }: { userId: string }) {
  const { mutate: deleteUser, isLoading } = useMutation(
    (id: string) => api.deleteUser(id)
  )

  const handleDelete = async () => {
    await deleteUser(userId)
    toast.success('User deleted')
  }

  return (
    <Button onClick={handleDelete} disabled={isLoading}>
      {isLoading ? 'Deleting...' : 'Delete'}
    </Button>
  )
}
```

### Local Storage Hook

```typescript
// hooks/useLocalStorage.ts
export function useLocalStorage<T>(
  key: string,
  initialValue: T
): [T, (value: T | ((prev: T) => T)) => void] {
  const [storedValue, setStoredValue] = useState<T>(() => {
    try {
      const item = window.localStorage.getItem(key)
      return item ? JSON.parse(item) : initialValue
    } catch {
      return initialValue
    }
  })

  const setValue = useCallback((value: T | ((prev: T) => T)) => {
    setStoredValue(prev => {
      const valueToStore = value instanceof Function ? value(prev) : value
      window.localStorage.setItem(key, JSON.stringify(valueToStore))
      return valueToStore
    })
  }, [key])

  return [storedValue, setValue]
}
```

### Debounce Hook

```typescript
// hooks/useDebounce.ts
export function useDebounce<T>(value: T, delay: number): T {
  const [debouncedValue, setDebouncedValue] = useState(value)

  useEffect(() => {
    const timer = setTimeout(() => setDebouncedValue(value), delay)
    return () => clearTimeout(timer)
  }, [value, delay])

  return debouncedValue
}

// hooks/useDebouncedCallback.ts
export function useDebouncedCallback<T extends (...args: any[]) => void>(
  callback: T,
  delay: number
): T {
  const callbackRef = useRef(callback)
  callbackRef.current = callback

  return useCallback(
    ((...args) => {
      const timer = setTimeout(() => callbackRef.current(...args), delay)
      return () => clearTimeout(timer)
    }) as T,
    [delay]
  )
}
```

### Toggle Hook

```typescript
// hooks/useToggle.ts
export function useToggle(initialValue = false) {
  const [value, setValue] = useState(initialValue)

  const toggle = useCallback(() => setValue(v => !v), [])
  const setTrue = useCallback(() => setValue(true), [])
  const setFalse = useCallback(() => setValue(false), [])

  return { value, toggle, setTrue, setFalse, setValue }
}

// Использование
function Modal() {
  const { value: isOpen, setTrue: open, setFalse: close } = useToggle()

  return (
    <>
      <Button onClick={open}>Open Modal</Button>
      {isOpen && <ModalContent onClose={close} />}
    </>
  )
}
```

### Previous Value Hook

```typescript
// hooks/usePrevious.ts
export function usePrevious<T>(value: T): T | undefined {
  const ref = useRef<T>()

  useEffect(() => {
    ref.current = value
  }, [value])

  return ref.current
}
```

### Click Outside Hook

```typescript
// hooks/useClickOutside.ts
export function useClickOutside<T extends HTMLElement>(
  handler: () => void
): RefObject<T> {
  const ref = useRef<T>(null)

  useEffect(() => {
    const listener = (event: MouseEvent | TouchEvent) => {
      if (!ref.current || ref.current.contains(event.target as Node)) {
        return
      }
      handler()
    }

    document.addEventListener('mousedown', listener)
    document.addEventListener('touchstart', listener)

    return () => {
      document.removeEventListener('mousedown', listener)
      document.removeEventListener('touchstart', listener)
    }
  }, [handler])

  return ref
}

// Использование
function Dropdown() {
  const { value: isOpen, setFalse: close } = useToggle()
  const ref = useClickOutside<HTMLDivElement>(close)

  return (
    <div ref={ref}>
      {isOpen && <DropdownMenu />}
    </div>
  )
}
```

## Feature-Specific Hooks

```typescript
// features/auth/hooks/useAuth.ts
export function useAuth() {
  const [user, setUser] = useState<User | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    authService.getCurrentUser()
      .then(setUser)
      .finally(() => setIsLoading(false))
  }, [])

  const login = useCallback(async (credentials: LoginCredentials) => {
    const user = await authService.login(credentials)
    setUser(user)
    return user
  }, [])

  const logout = useCallback(async () => {
    await authService.logout()
    setUser(null)
  }, [])

  return {
    user,
    isLoading,
    isAuthenticated: !!user,
    login,
    logout,
  }
}
```

## Правила именования

```typescript
// ✅ Хорошо
useUser()           // Получение данных
useUsers()          // Получение списка
useCreateUser()     // Создание
useUpdateUser()     // Обновление
useDeleteUser()     // Удаление
useUserForm()       // Форма
useUserSearch()     // Поиск

// ❌ Плохо
useData()           // Слишком общее
useStuff()          // Непонятно
useHandleClick()    // Это не хук, а колбэк
```

## Композиция хуков

```typescript
// Комбинируй простые хуки в сложные
function useUserProfile(userId: string) {
  const { data: user, isLoading: userLoading } = useUser(userId)
  const { data: posts, isLoading: postsLoading } = useUserPosts(userId)
  const { mutate: updateUser } = useUpdateUser()

  return {
    user,
    posts,
    isLoading: userLoading || postsLoading,
    updateUser,
  }
}
```
