# Производительность

Оптимизация React приложений.

## Мемоизация

### useMemo — кэширование вычислений

```tsx
// ✅ Используй для дорогих вычислений
function ProductList({ products, filter }: Props) {
  const filteredProducts = useMemo(
    () => products.filter(p => p.category === filter),
    [products, filter]
  )

  return <List items={filteredProducts} />
}

// ❌ НЕ используй для простых операций
const name = useMemo(() => `${firstName} ${lastName}`, [firstName, lastName])
// Просто напиши:
const name = `${firstName} ${lastName}`
```

### useCallback — кэширование функций

```tsx
// ✅ Когда передаёшь колбэк в мемоизированный компонент
function Parent() {
  const [count, setCount] = useState(0)

  const handleClick = useCallback(() => {
    console.log('clicked')
  }, [])

  return <MemoizedChild onClick={handleClick} />
}

// ✅ Когда функция в зависимостях useEffect
function SearchInput({ onSearch }: Props) {
  const [query, setQuery] = useState('')

  const debouncedSearch = useCallback(
    debounce((q: string) => onSearch(q), 300),
    [onSearch]
  )

  useEffect(() => {
    debouncedSearch(query)
  }, [query, debouncedSearch])
}
```

### React.memo — предотвращение ререндеров

```tsx
// ✅ Для компонентов с дорогим рендерингом
const ExpensiveList = memo(function ExpensiveList({ items }: Props) {
  return (
    <ul>
      {items.map(item => (
        <ExpensiveItem key={item.id} item={item} />
      ))}
    </ul>
  )
})

// ✅ С кастомным сравнением
const UserCard = memo(
  function UserCard({ user }: { user: User }) {
    return <div>{user.name}</div>
  },
  (prevProps, nextProps) => prevProps.user.id === nextProps.user.id
)
```

## Когда НЕ мемоизировать

```tsx
// ❌ Примитивные пропсы
<Button onClick={useCallback(() => setOpen(true), [])} />
// Если Button не обёрнут в memo — useCallback бесполезен

// ❌ Новые объекты каждый рендер всё равно
<List style={useMemo(() => ({ color: 'red' }), [])} />
// Проще вынести константу:
const style = { color: 'red' }
<List style={style} />

// ❌ Компонент и так быстрый
const SimpleText = memo(({ text }) => <span>{text}</span>)
// Overhead memo > benefit
```

## Виртуализация списков

```tsx
// react-window для больших списков
import { FixedSizeList } from 'react-window'

function VirtualList({ items }: { items: Item[] }) {
  return (
    <FixedSizeList
      height={400}
      itemCount={items.length}
      itemSize={50}
      width="100%"
    >
      {({ index, style }) => (
        <div style={style}>
          <ItemRow item={items[index]} />
        </div>
      )}
    </FixedSizeList>
  )
}
```

## Code Splitting

```tsx
// Lazy loading компонентов
import { lazy, Suspense } from 'react'

const Dashboard = lazy(() => import('./pages/Dashboard'))
const Settings = lazy(() => import('./pages/Settings'))

function App() {
  return (
    <Suspense fallback={<PageLoader />}>
      <Routes>
        <Route path="/dashboard" element={<Dashboard />} />
        <Route path="/settings" element={<Settings />} />
      </Routes>
    </Suspense>
  )
}
```

## Избегай лишних ререндеров

### 1. Поднятие состояния

```tsx
// ❌ Плохо — весь список ререндерится при изменении input
function SearchableList() {
  const [query, setQuery] = useState('')
  const [items] = useState(largeArray)

  return (
    <div>
      <input value={query} onChange={e => setQuery(e.target.value)} />
      <List items={items.filter(i => i.includes(query))} />
    </div>
  )
}

// ✅ Хорошо — input изолирован
function SearchableList() {
  const [items] = useState(largeArray)
  return <FilteredList items={items} />
}

function FilteredList({ items }: Props) {
  const [query, setQuery] = useState('')
  const filtered = useMemo(
    () => items.filter(i => i.includes(query)),
    [items, query]
  )

  return (
    <div>
      <SearchInput value={query} onChange={setQuery} />
      <List items={filtered} />
    </div>
  )
}
```

### 2. Children паттерн

```tsx
// ❌ Плохо — children ререндерятся при изменении state
function Parent() {
  const [count, setCount] = useState(0)
  return (
    <div>
      <button onClick={() => setCount(c => c + 1)}>{count}</button>
      <ExpensiveChild />
    </div>
  )
}

// ✅ Хорошо — children передаются как props
function Parent({ children }: PropsWithChildren) {
  const [count, setCount] = useState(0)
  return (
    <div>
      <button onClick={() => setCount(c => c + 1)}>{count}</button>
      {children}
    </div>
  )
}

// Использование
<Parent>
  <ExpensiveChild />
</Parent>
```

### 3. Key для сброса состояния

```tsx
// Сброс формы при смене userId
<UserForm key={userId} userId={userId} />
```

## Оптимизация изображений

```tsx
// Lazy loading изображений
function LazyImage({ src, alt }: Props) {
  return (
    <img
      src={src}
      alt={alt}
      loading="lazy"
      decoding="async"
    />
  )
}

// С placeholder
function ImageWithPlaceholder({ src, alt }: Props) {
  const [loaded, setLoaded] = useState(false)

  return (
    <div className="relative">
      {!loaded && <Skeleton className="absolute inset-0" />}
      <img
        src={src}
        alt={alt}
        loading="lazy"
        onLoad={() => setLoaded(true)}
        className={loaded ? 'opacity-100' : 'opacity-0'}
      />
    </div>
  )
}
```

## Профилирование

```tsx
// React DevTools Profiler
import { Profiler } from 'react'

function onRenderCallback(
  id: string,
  phase: 'mount' | 'update',
  actualDuration: number
) {
  console.log({ id, phase, actualDuration })
}

<Profiler id="Navigation" onRender={onRenderCallback}>
  <Navigation />
</Profiler>
```

## Чеклист производительности

- [ ] memo() для тяжёлых компонентов
- [ ] useMemo для дорогих вычислений
- [ ] useCallback для колбэков в memo-компонентах
- [ ] Виртуализация для списков > 100 элементов
- [ ] Code splitting для роутов
- [ ] Lazy loading изображений
- [ ] Избегание inline объектов/функций в JSX
