# State Management

Zustand для клиентского состояния + TanStack Query для серверного.

## Философия

```
┌─────────────────────────────────────────────────────────┐
│                    Application State                     │
├───────────────────────────┬─────────────────────────────┤
│      Client State         │       Server State          │
│       (Zustand)           │    (TanStack Query)         │
├───────────────────────────┼─────────────────────────────┤
│ • UI state (modals)       │ • API data                  │
│ • User preferences        │ • Cached responses          │
│ • Form drafts             │ • Background refetch        │
│ • Filters, sorting        │ • Optimistic updates        │
│ • Theme                   │ • Pagination                │
└───────────────────────────┴─────────────────────────────┘
```

---

# Zustand

Лёгкий state manager для клиентского состояния.

## Установка

```bash
npm install zustand
```

## Базовый Store

```typescript
// stores/useCounterStore.ts
import { create } from 'zustand'

interface CounterState {
  count: number
  increment: () => void
  decrement: () => void
  reset: () => void
}

export const useCounterStore = create<CounterState>((set) => ({
  count: 0,
  increment: () => set((state) => ({ count: state.count + 1 })),
  decrement: () => set((state) => ({ count: state.count - 1 })),
  reset: () => set({ count: 0 }),
}))

// Использование в компоненте
function Counter() {
  const { count, increment, decrement } = useCounterStore()

  return (
    <div>
      <span>{count}</span>
      <button onClick={increment}>+</button>
      <button onClick={decrement}>-</button>
    </div>
  )
}
```

## Store с async actions

```typescript
// stores/useUserStore.ts
import { create } from 'zustand'
import { api } from '@/lib/api'

interface User {
  id: string
  name: string
  email: string
}

interface UserState {
  user: User | null
  isLoading: boolean
  error: string | null
  login: (email: string, password: string) => Promise<void>
  logout: () => void
  setUser: (user: User | null) => void
}

export const useUserStore = create<UserState>((set, get) => ({
  user: null,
  isLoading: false,
  error: null,

  login: async (email, password) => {
    set({ isLoading: true, error: null })
    try {
      const user = await api.login(email, password)
      set({ user, isLoading: false })
    } catch (error) {
      set({ error: (error as Error).message, isLoading: false })
      throw error
    }
  },

  logout: () => {
    api.logout()
    set({ user: null })
  },

  setUser: (user) => set({ user }),
}))
```

## Store с Persist middleware

```typescript
// stores/useSettingsStore.ts
import { create } from 'zustand'
import { persist, createJSONStorage } from 'zustand/middleware'

interface SettingsState {
  theme: 'light' | 'dark' | 'system'
  language: string
  notifications: boolean
  setTheme: (theme: SettingsState['theme']) => void
  setLanguage: (language: string) => void
  toggleNotifications: () => void
}

export const useSettingsStore = create<SettingsState>()(
  persist(
    (set) => ({
      theme: 'system',
      language: 'ru',
      notifications: true,

      setTheme: (theme) => set({ theme }),
      setLanguage: (language) => set({ language }),
      toggleNotifications: () => set((state) => ({
        notifications: !state.notifications
      })),
    }),
    {
      name: 'settings-storage',
      storage: createJSONStorage(() => localStorage),
      // Выбираем что персистить
      partialize: (state) => ({
        theme: state.theme,
        language: state.language,
        notifications: state.notifications,
      }),
    }
  )
)
```

## Slices Pattern (для больших store)

```typescript
// stores/slices/userSlice.ts
import { StateCreator } from 'zustand'

export interface UserSlice {
  user: User | null
  setUser: (user: User | null) => void
}

export const createUserSlice: StateCreator<UserSlice> = (set) => ({
  user: null,
  setUser: (user) => set({ user }),
})

// stores/slices/cartSlice.ts
export interface CartSlice {
  items: CartItem[]
  addItem: (item: CartItem) => void
  removeItem: (id: string) => void
  clearCart: () => void
}

export const createCartSlice: StateCreator<CartSlice> = (set) => ({
  items: [],
  addItem: (item) => set((state) => ({ items: [...state.items, item] })),
  removeItem: (id) => set((state) => ({
    items: state.items.filter((i) => i.id !== id)
  })),
  clearCart: () => set({ items: [] }),
})

// stores/useAppStore.ts
import { create } from 'zustand'
import { createUserSlice, UserSlice } from './slices/userSlice'
import { createCartSlice, CartSlice } from './slices/cartSlice'

type AppStore = UserSlice & CartSlice

export const useAppStore = create<AppStore>()((...a) => ({
  ...createUserSlice(...a),
  ...createCartSlice(...a),
}))
```

## Селекторы (оптимизация ререндеров)

```typescript
// ❌ Плохо — подписка на весь store
function Header() {
  const store = useUserStore() // Ререндер при любом изменении
  return <span>{store.user?.name}</span>
}

// ✅ Хорошо — селектор на нужное поле
function Header() {
  const userName = useUserStore((state) => state.user?.name)
  return <span>{userName}</span>
}

// ✅ Несколько полей с shallow compare
import { useShallow } from 'zustand/react/shallow'

function UserInfo() {
  const { name, email } = useUserStore(
    useShallow((state) => ({
      name: state.user?.name,
      email: state.user?.email
    }))
  )
  return <div>{name} - {email}</div>
}
```

---

# TanStack Query

Для серверного состояния (API данные).

## Установка

```bash
npm install @tanstack/react-query
```

## Setup

```typescript
// app/providers/QueryProvider.tsx
import { QueryClient, QueryClientProvider } from '@tanstack/react-query'
import { ReactQueryDevtools } from '@tanstack/react-query-devtools'

const queryClient = new QueryClient({
  defaultOptions: {
    queries: {
      staleTime: 1000 * 60 * 5, // 5 минут
      gcTime: 1000 * 60 * 30,   // 30 минут
      retry: 1,
      refetchOnWindowFocus: false,
    },
  },
})

export function QueryProvider({ children }: { children: React.ReactNode }) {
  return (
    <QueryClientProvider client={queryClient}>
      {children}
      <ReactQueryDevtools initialIsOpen={false} />
    </QueryClientProvider>
  )
}
```

## useQuery — получение данных

```typescript
// features/users/hooks/useUser.ts
import { useQuery } from '@tanstack/react-query'
import { api } from '@/lib/api'

export function useUser(userId: string) {
  return useQuery({
    queryKey: ['user', userId],
    queryFn: () => api.getUser(userId),
    enabled: !!userId, // Не запускать без userId
  })
}

// features/users/hooks/useUsers.ts
export function useUsers(filters?: UserFilters) {
  return useQuery({
    queryKey: ['users', filters],
    queryFn: () => api.getUsers(filters),
    placeholderData: (previousData) => previousData, // Keep old data while fetching
  })
}

// Использование
function UserProfile({ userId }: { userId: string }) {
  const { data: user, isLoading, error, refetch } = useUser(userId)

  if (isLoading) return <Spinner />
  if (error) return <ErrorMessage error={error} onRetry={refetch} />

  return <UserCard user={user!} />
}
```

## useMutation — изменение данных

```typescript
// features/users/hooks/useCreateUser.ts
import { useMutation, useQueryClient } from '@tanstack/react-query'
import { api } from '@/lib/api'

export function useCreateUser() {
  const queryClient = useQueryClient()

  return useMutation({
    mutationFn: (data: CreateUserData) => api.createUser(data),
    onSuccess: () => {
      // Инвалидация списка пользователей
      queryClient.invalidateQueries({ queryKey: ['users'] })
    },
  })
}

// features/users/hooks/useUpdateUser.ts
export function useUpdateUser() {
  const queryClient = useQueryClient()

  return useMutation({
    mutationFn: ({ id, data }: { id: string; data: UpdateUserData }) =>
      api.updateUser(id, data),
    onSuccess: (updatedUser) => {
      // Обновление кэша конкретного пользователя
      queryClient.setQueryData(['user', updatedUser.id], updatedUser)
      // Инвалидация списка
      queryClient.invalidateQueries({ queryKey: ['users'] })
    },
  })
}

// Использование
function CreateUserForm() {
  const { mutate, isPending, error } = useCreateUser()

  const handleSubmit = (data: CreateUserData) => {
    mutate(data, {
      onSuccess: () => {
        toast.success('User created!')
      },
      onError: (error) => {
        toast.error(error.message)
      },
    })
  }

  return <Form onSubmit={handleSubmit} isLoading={isPending} error={error} />
}
```

## Optimistic Updates

```typescript
export function useToggleTodo() {
  const queryClient = useQueryClient()

  return useMutation({
    mutationFn: (todoId: string) => api.toggleTodo(todoId),

    onMutate: async (todoId) => {
      // Отменяем исходящие запросы
      await queryClient.cancelQueries({ queryKey: ['todos'] })

      // Сохраняем предыдущее состояние
      const previousTodos = queryClient.getQueryData(['todos'])

      // Оптимистично обновляем
      queryClient.setQueryData(['todos'], (old: Todo[]) =>
        old.map((todo) =>
          todo.id === todoId ? { ...todo, done: !todo.done } : todo
        )
      )

      return { previousTodos }
    },

    onError: (err, todoId, context) => {
      // Откат при ошибке
      queryClient.setQueryData(['todos'], context?.previousTodos)
    },

    onSettled: () => {
      // Обновляем данные в любом случае
      queryClient.invalidateQueries({ queryKey: ['todos'] })
    },
  })
}
```

## Query Keys организация

```typescript
// lib/queryKeys.ts
export const queryKeys = {
  users: {
    all: ['users'] as const,
    lists: () => [...queryKeys.users.all, 'list'] as const,
    list: (filters: UserFilters) => [...queryKeys.users.lists(), filters] as const,
    details: () => [...queryKeys.users.all, 'detail'] as const,
    detail: (id: string) => [...queryKeys.users.details(), id] as const,
  },
  todos: {
    all: ['todos'] as const,
    list: (filters?: TodoFilters) => [...queryKeys.todos.all, filters] as const,
    detail: (id: string) => [...queryKeys.todos.all, id] as const,
  },
}

// Использование
useQuery({
  queryKey: queryKeys.users.detail(userId),
  queryFn: () => api.getUser(userId),
})

// Инвалидация всех user queries
queryClient.invalidateQueries({ queryKey: queryKeys.users.all })
```

---

## Комбинирование Zustand + TanStack Query

```typescript
// Zustand — UI состояние
const useUIStore = create((set) => ({
  isModalOpen: false,
  selectedUserId: null,
  openModal: (userId) => set({ isModalOpen: true, selectedUserId: userId }),
  closeModal: () => set({ isModalOpen: false, selectedUserId: null }),
}))

// TanStack Query — данные
function UserModal() {
  const { isModalOpen, selectedUserId, closeModal } = useUIStore()
  const { data: user, isLoading } = useUser(selectedUserId!)

  if (!isModalOpen) return null

  return (
    <Modal onClose={closeModal}>
      {isLoading ? <Spinner /> : <UserDetails user={user!} />}
    </Modal>
  )
}
```

## Чеклист

- [ ] Zustand для UI/клиентского состояния
- [ ] TanStack Query для API данных
- [ ] Селекторы в Zustand для оптимизации
- [ ] Query keys организованы
- [ ] Optimistic updates для UX
- [ ] Persist для настроек пользователя
