# Unit тестирование

Vitest + React Testing Library для тестирования компонентов.

## Настройка

```bash
npm install -D vitest @testing-library/react @testing-library/user-event @testing-library/jest-dom jsdom
```

```typescript
// vite.config.ts
import { defineConfig } from 'vite'
import react from '@vitejs/plugin-react'

export default defineConfig({
  plugins: [react()],
  test: {
    globals: true,
    environment: 'jsdom',
    setupFiles: './src/test/setup.ts',
    css: true,
  },
})
```

```typescript
// src/test/setup.ts
import '@testing-library/jest-dom'
import { cleanup } from '@testing-library/react'
import { afterEach } from 'vitest'

// Cleanup after each test
afterEach(() => {
  cleanup()
})
```

## Базовый тест компонента

```tsx
// components/ui/Button/Button.test.tsx
import { render, screen } from '@testing-library/react'
import userEvent from '@testing-library/user-event'
import { describe, it, expect, vi } from 'vitest'
import { Button } from './Button'

describe('Button', () => {
  it('renders children correctly', () => {
    render(<Button>Click me</Button>)
    expect(screen.getByRole('button', { name: /click me/i })).toBeInTheDocument()
  })

  it('calls onClick when clicked', async () => {
    const user = userEvent.setup()
    const handleClick = vi.fn()

    render(<Button onClick={handleClick}>Click</Button>)
    await user.click(screen.getByRole('button'))

    expect(handleClick).toHaveBeenCalledTimes(1)
  })

  it('is disabled when isLoading is true', () => {
    render(<Button isLoading>Submit</Button>)
    expect(screen.getByRole('button')).toBeDisabled()
  })

  it('applies variant styles correctly', () => {
    render(<Button variant="danger">Delete</Button>)
    expect(screen.getByRole('button')).toHaveClass('bg-red-600')
  })
})
```

## Тестирование с async/await

```tsx
// features/auth/components/LoginForm.test.tsx
import { render, screen, waitFor } from '@testing-library/react'
import userEvent from '@testing-library/user-event'
import { describe, it, expect, vi } from 'vitest'
import { LoginForm } from './LoginForm'

describe('LoginForm', () => {
  it('submits form with valid credentials', async () => {
    const user = userEvent.setup()
    const onSubmit = vi.fn()

    render(<LoginForm onSubmit={onSubmit} />)

    await user.type(screen.getByLabelText(/email/i), 'test@example.com')
    await user.type(screen.getByLabelText(/password/i), 'password123')
    await user.click(screen.getByRole('button', { name: /sign in/i }))

    await waitFor(() => {
      expect(onSubmit).toHaveBeenCalledWith({
        email: 'test@example.com',
        password: 'password123',
      })
    })
  })

  it('shows validation errors for empty fields', async () => {
    const user = userEvent.setup()

    render(<LoginForm onSubmit={vi.fn()} />)
    await user.click(screen.getByRole('button', { name: /sign in/i }))

    expect(await screen.findByText(/email is required/i)).toBeInTheDocument()
    expect(await screen.findByText(/password is required/i)).toBeInTheDocument()
  })

  it('shows loading state while submitting', async () => {
    const user = userEvent.setup()
    const onSubmit = vi.fn(() => new Promise(resolve => setTimeout(resolve, 100)))

    render(<LoginForm onSubmit={onSubmit} />)

    await user.type(screen.getByLabelText(/email/i), 'test@example.com')
    await user.type(screen.getByLabelText(/password/i), 'password123')
    await user.click(screen.getByRole('button', { name: /sign in/i }))

    expect(screen.getByRole('button')).toBeDisabled()
    expect(screen.getByText(/signing in/i)).toBeInTheDocument()
  })
})
```

## Тестирование хуков

```tsx
// hooks/useCounter.test.ts
import { renderHook, act } from '@testing-library/react'
import { describe, it, expect } from 'vitest'
import { useCounter } from './useCounter'

describe('useCounter', () => {
  it('initializes with default value', () => {
    const { result } = renderHook(() => useCounter())
    expect(result.current.count).toBe(0)
  })

  it('initializes with provided value', () => {
    const { result } = renderHook(() => useCounter(10))
    expect(result.current.count).toBe(10)
  })

  it('increments counter', () => {
    const { result } = renderHook(() => useCounter())

    act(() => {
      result.current.increment()
    })

    expect(result.current.count).toBe(1)
  })

  it('decrements counter', () => {
    const { result } = renderHook(() => useCounter(5))

    act(() => {
      result.current.decrement()
    })

    expect(result.current.count).toBe(4)
  })
})
```

## Тестирование с Context

```tsx
// test/utils.tsx — Custom render с providers
import { render, RenderOptions } from '@testing-library/react'
import { ReactElement } from 'react'
import { AuthProvider } from '@/app/providers/AuthProvider'
import { ThemeProvider } from '@/app/providers/ThemeProvider'

const AllProviders = ({ children }: { children: React.ReactNode }) => {
  return (
    <AuthProvider>
      <ThemeProvider>
        {children}
      </ThemeProvider>
    </AuthProvider>
  )
}

const customRender = (
  ui: ReactElement,
  options?: Omit<RenderOptions, 'wrapper'>
) => render(ui, { wrapper: AllProviders, ...options })

export * from '@testing-library/react'
export { customRender as render }
```

```tsx
// Использование
import { render, screen } from '@/test/utils'
import { UserProfile } from './UserProfile'

it('shows user info from context', () => {
  render(<UserProfile />)
  expect(screen.getByText(/logged in as/i)).toBeInTheDocument()
})
```

## Мокирование API

```tsx
// С MSW (рекомендуется)
import { http, HttpResponse } from 'msw'
import { setupServer } from 'msw/node'

const server = setupServer(
  http.get('/api/users/:id', ({ params }) => {
    return HttpResponse.json({
      id: params.id,
      name: 'John Doe',
      email: 'john@example.com',
    })
  })
)

beforeAll(() => server.listen())
afterEach(() => server.resetHandlers())
afterAll(() => server.close())

it('fetches and displays user', async () => {
  render(<UserProfile userId="123" />)

  expect(await screen.findByText('John Doe')).toBeInTheDocument()
  expect(screen.getByText('john@example.com')).toBeInTheDocument()
})

it('handles error state', async () => {
  server.use(
    http.get('/api/users/:id', () => {
      return HttpResponse.json({ error: 'Not found' }, { status: 404 })
    })
  )

  render(<UserProfile userId="999" />)

  expect(await screen.findByText(/user not found/i)).toBeInTheDocument()
})
```

## Query приоритеты

```tsx
// По приоритету (от лучшего к худшему)
screen.getByRole('button', { name: /submit/i })     // 1. Роли (accessibility)
screen.getByLabelText(/email/i)                      // 2. Label (формы)
screen.getByPlaceholderText(/search/i)               // 3. Placeholder
screen.getByText(/welcome/i)                         // 4. Текст
screen.getByDisplayValue('john@example.com')         // 5. Значение input
screen.getByAltText(/profile/i)                      // 6. Alt text
screen.getByTitle(/close/i)                          // 7. Title
screen.getByTestId('custom-element')                 // 8. Test ID (последний resort)
```

## Assertions

```tsx
// Наличие
expect(element).toBeInTheDocument()
expect(element).not.toBeInTheDocument()

// Видимость
expect(element).toBeVisible()
expect(element).not.toBeVisible()

// Состояние
expect(button).toBeDisabled()
expect(button).toBeEnabled()
expect(input).toBeRequired()
expect(checkbox).toBeChecked()

// Содержимое
expect(element).toHaveTextContent(/hello/i)
expect(element).toHaveValue('test')
expect(element).toHaveAttribute('href', '/home')

// Классы
expect(element).toHaveClass('active')

// Фокус
expect(input).toHaveFocus()
```

## Структура тестов

```tsx
describe('ComponentName', () => {
  // Setup общий для всех тестов
  const defaultProps = {
    onSubmit: vi.fn(),
  }

  beforeEach(() => {
    vi.clearAllMocks()
  })

  describe('rendering', () => {
    it('renders correctly with required props', () => {})
    it('renders optional elements when provided', () => {})
  })

  describe('user interactions', () => {
    it('calls onSubmit when form is submitted', async () => {})
    it('validates input on blur', async () => {})
  })

  describe('edge cases', () => {
    it('handles empty data', () => {})
    it('handles error state', () => {})
  })
})
```

## Файловая структура

```
Button/
├── Button.tsx
├── Button.test.tsx    # Unit тест рядом с компонентом
└── index.ts
```
