# API Контроллеры

Тонкие контроллеры — только маршрутизация, делегирование сервисам, рендеринг.

## Базовый контроллер

```ruby
# app/controllers/api/v1/application_controller.rb
module Api
  module V1
    class ApplicationController < ActionController::API
      include ActionController::MimeResponds

      rescue_from ActiveRecord::RecordNotFound, with: :not_found
      rescue_from ActionController::ParameterMissing, with: :bad_request
      rescue_from ActiveRecord::RecordInvalid, with: :unprocessable

      private

      def not_found(exception)
        render_error('Not Found', exception.message, :not_found)
      end

      def bad_request(exception)
        render_error('Bad Request', exception.message, :bad_request)
      end

      def unprocessable(exception)
        render_error('Validation Error', exception.message, :unprocessable_entity)
      end

      def render_error(title, detail, status)
        render json: {
          errors: [{ title: title, detail: detail }]
        }, status: status
      end

      def render_errors(errors, status: :unprocessable_entity)
        formatted = Array(errors).map { |e| { detail: e } }
        render json: { errors: formatted }, status: status
      end

      def render_resource(resource, serializer:, status: :ok, **options)
        render json: serializer.new(resource, options).serializable_hash, status: status
      end

      def render_collection(collection, serializer:, **options)
        meta = build_pagination_meta(collection) if collection.respond_to?(:total_count)
        render_resource(collection, serializer: serializer, meta: meta, **options)
      end

      def build_pagination_meta(collection)
        {
          total_count: collection.total_count,
          total_pages: collection.total_pages,
          current_page: collection.current_page,
          per_page: collection.limit_value
        }
      end
    end
  end
end
```

## Пример ресурсного контроллера

```ruby
# app/controllers/api/v1/users_controller.rb
module Api
  module V1
    class UsersController < ApplicationController
      before_action :set_user, only: %i[show update destroy]

      # GET /api/v1/users
      def index
        users = Users::ListQuery.call(params: filter_params)
        render_collection(users, serializer: UserSerializer)
      end

      # GET /api/v1/users/:id
      def show
        render_resource(@user, serializer: UserSerializer)
      end

      # POST /api/v1/users
      def create
        result = Users::CreateService.call(params: user_params)

        if result.success?
          render_resource(result.data, serializer: UserSerializer, status: :created)
        else
          render_errors(result.errors)
        end
      end

      # PATCH/PUT /api/v1/users/:id
      def update
        result = Users::UpdateService.call(user: @user, params: user_params)

        if result.success?
          render_resource(result.data, serializer: UserSerializer)
        else
          render_errors(result.errors)
        end
      end

      # DELETE /api/v1/users/:id
      def destroy
        result = Users::DestroyService.call(user: @user)

        if result.success?
          head :no_content
        else
          render_errors(result.errors)
        end
      end

      private

      def set_user
        @user = User.find(params[:id])
      end

      def user_params
        params.require(:user).permit(:email, :name, :password, :role)
      end

      def filter_params
        params.permit(:status, :role, :search, :page, :per_page, :sort)
      end
    end
  end
end
```

## Контроллер с аутентификацией

```ruby
module Api
  module V1
    class ApplicationController < ActionController::API
      before_action :authenticate_user!

      private

      def authenticate_user!
        @current_user = authenticate_with_token
        render_error('Unauthorized', 'Invalid or missing token', :unauthorized) unless @current_user
      end

      def authenticate_with_token
        token = request.headers['Authorization']&.split(' ')&.last
        return unless token

        Users::AuthenticateTokenService.call(token: token).data
      end

      attr_reader :current_user
    end
  end
end
```

## Вложенные ресурсы

```ruby
# config/routes.rb
namespace :api do
  namespace :v1 do
    resources :users do
      resources :posts, only: %i[index create]
    end
    resources :posts, only: %i[show update destroy]
  end
end

# app/controllers/api/v1/posts_controller.rb
module Api
  module V1
    class PostsController < ApplicationController
      before_action :set_user, only: %i[index create]
      before_action :set_post, only: %i[show update destroy]

      # GET /api/v1/users/:user_id/posts
      def index
        posts = @user.posts.order(created_at: :desc)
        render_collection(posts, serializer: PostSerializer)
      end

      # POST /api/v1/users/:user_id/posts
      def create
        result = Posts::CreateService.call(user: @user, params: post_params)

        if result.success?
          render_resource(result.data, serializer: PostSerializer, status: :created)
        else
          render_errors(result.errors)
        end
      end

      # GET /api/v1/posts/:id
      def show
        render_resource(@post, serializer: PostSerializer)
      end

      private

      def set_user
        @user = User.find(params[:user_id])
      end

      def set_post
        @post = Post.find(params[:id])
      end

      def post_params
        params.require(:post).permit(:title, :body, :published)
      end
    end
  end
end
```

## Правила

1. **Только 7 RESTful actions** — index, show, create, update, destroy, new, edit
2. **Нестандартные действия** — выноси в отдельный контроллер
3. **Никакой логики** — только вызов сервисов
4. **Strong parameters** — всегда через permit
5. **Before actions** — для set_* методов
6. **Consistent responses** — через render_resource/render_errors

## Нестандартные действия

```ruby
# Вместо users#activate
# Создай отдельный контроллер

# app/controllers/api/v1/user_activations_controller.rb
module Api
  module V1
    class UserActivationsController < ApplicationController
      # POST /api/v1/users/:user_id/activation
      def create
        user = User.find(params[:user_id])
        result = Users::ActivateService.call(user: user)

        if result.success?
          render_resource(result.data, serializer: UserSerializer)
        else
          render_errors(result.errors)
        end
      end

      # DELETE /api/v1/users/:user_id/activation
      def destroy
        user = User.find(params[:user_id])
        result = Users::DeactivateService.call(user: user)

        if result.success?
          head :no_content
        else
          render_errors(result.errors)
        end
      end
    end
  end
end
```
