# Draper Декораторы

Презентационная логика в декораторах, не в моделях и не во views.

## Установка

```ruby
gem 'draper'
```

```bash
rails g decorator User
```

## Базовый декоратор

```ruby
# app/decorators/application_decorator.rb
class ApplicationDecorator < Draper::Decorator
  delegate_all

  def formatted_date(date, format = '%d.%m.%Y')
    date&.strftime(format)
  end

  def formatted_datetime(datetime)
    datetime&.strftime('%d.%m.%Y %H:%M')
  end

  def formatted_money(amount, currency = '₽')
    return '—' if amount.nil?
    "#{h.number_with_delimiter(amount, delimiter: ' ')} #{currency}"
  end
end
```

## Пример декоратора

```ruby
# app/decorators/user_decorator.rb
class UserDecorator < ApplicationDecorator
  delegate_all

  def full_name
    "#{first_name} #{last_name}".strip
  end

  def display_name
    full_name.presence || email.split('@').first
  end

  def formatted_created_at
    formatted_datetime(created_at)
  end

  def status_badge
    case status
    when 'active'
      { text: 'Активен', color: 'green', icon: 'check' }
    when 'pending'
      { text: 'Ожидает', color: 'yellow', icon: 'clock' }
    when 'suspended'
      { text: 'Заблокирован', color: 'red', icon: 'ban' }
    else
      { text: status.humanize, color: 'gray', icon: 'question' }
    end
  end

  def avatar_url
    if object.avatar.attached?
      h.url_for(object.avatar)
    else
      h.asset_path('default_avatar.png')
    end
  end

  def role_label
    I18n.t("users.roles.#{role}", default: role.humanize)
  end

  def short_bio(length: 100)
    h.truncate(bio.to_s, length: length)
  end

  # Для JSON API (если нужно)
  def as_json_summary
    {
      id: id,
      name: display_name,
      avatar: avatar_url,
      status: status_badge
    }
  end
end
```

## Использование

```ruby
# Один объект
@user = User.find(params[:id]).decorate
@user.full_name
@user.status_badge

# Коллекция
@users = UserDecorator.decorate_collection(User.active)

# Или через scope
@users = User.active.decorate

# С контекстом
@user = user.decorate(context: { current_user: current_user })

# В декораторе доступ к контексту
def editable?
  context[:current_user]&.admin? || context[:current_user] == object
end
```

## Декоратор с ассоциациями

```ruby
class PostDecorator < ApplicationDecorator
  delegate_all

  # Декорирует ассоциацию автоматически
  decorates_association :author
  decorates_association :comments

  # С кастомным декоратором
  decorates_association :category, with: CategoryDecorator

  def author_name
    author.display_name  # author уже задекорирован
  end

  def formatted_published_at
    return 'Черновик' unless published_at
    formatted_datetime(published_at)
  end

  def reading_time
    words = body.to_s.split.size
    minutes = (words / 200.0).ceil
    "#{minutes} мин"
  end
end
```

## Декоратор для коллекций

```ruby
class PaginatedCollectionDecorator < Draper::CollectionDecorator
  delegate :current_page, :total_pages, :limit_value, :total_count

  def page_info
    "Страница #{current_page} из #{total_pages}"
  end

  def showing_info
    from = (current_page - 1) * limit_value + 1
    to = [current_page * limit_value, total_count].min
    "Показано #{from}-#{to} из #{total_count}"
  end
end

# Использование
class PostDecorator < ApplicationDecorator
  def self.collection_decorator_class
    PaginatedCollectionDecorator
  end
end

@posts = PostDecorator.decorate_collection(Post.page(1))
@posts.page_info  # => "Страница 1 из 5"
```

## Finders в декораторе

```ruby
class UserDecorator < ApplicationDecorator
  decorates_finders

  # Теперь можно
  # UserDecorator.find(1)        => decorated user
  # UserDecorator.find_by(...)   => decorated user
  # UserDecorator.where(...)     => decorated collection
end
```

## Хелперы

```ruby
class UserDecorator < ApplicationDecorator
  # Доступ к Rails хелперам через h
  def profile_link
    h.link_to display_name, h.user_path(object)
  end

  def avatar_tag(size: 40)
    h.image_tag(avatar_url, size: "#{size}x#{size}", class: 'avatar')
  end

  # Доступ к routes
  def edit_url
    h.edit_user_url(object)
  end
end
```

## Когда использовать декораторы

✅ **Используй декораторы для:**
- Форматирование дат, чисел, денег
- Генерация display names
- Статусы и badges
- Truncation текста
- URL для изображений
- Локализация значений

❌ **НЕ используй для:**
- Бизнес-логика (→ сервисы)
- Валидация (→ модели)
- Авторизация (→ policies)
- Запросы к БД (→ модели/queries)
