# Модели

**В моделях НЕТ бизнес-логики!**

Модели содержат только:
- Associations
- Validations
- Scopes
- Enums
- Callbacks (минимально, только для данных)

## Структура модели

```ruby
# app/models/user.rb
class User < ApplicationRecord
  # === Associations ===
  belongs_to :organization, optional: true
  has_many :posts, dependent: :destroy
  has_many :comments, dependent: :destroy
  has_one :profile, dependent: :destroy

  # === Validations ===
  validates :email, presence: true,
                    uniqueness: { case_sensitive: false },
                    format: { with: URI::MailTo::EMAIL_REGEXP }
  validates :name, presence: true,
                   length: { minimum: 2, maximum: 100 }
  validates :password, length: { minimum: 8 }, allow_nil: true

  # === Enums ===
  enum :status, { pending: 0, active: 1, suspended: 2 }, prefix: true
  enum :role, { user: 0, moderator: 1, admin: 2 }, prefix: :is

  # === Scopes ===
  scope :active, -> { where(status: :active) }
  scope :admins, -> { where(role: :admin) }
  scope :recent, -> { order(created_at: :desc) }
  scope :created_after, ->(date) { where('created_at > ?', date) }
  scope :by_organization, ->(org_id) { where(organization_id: org_id) }
  scope :search, ->(query) {
    where('name ILIKE :q OR email ILIKE :q', q: "%#{query}%")
  }

  # === Callbacks (только для данных) ===
  before_save :normalize_email
  before_create :set_defaults

  private

  def normalize_email
    self.email = email.to_s.downcase.strip
  end

  def set_defaults
    self.status ||= :pending
    self.role ||= :user
  end
end
```

## Правила для Associations

```ruby
# Всегда указывай dependent
has_many :posts, dependent: :destroy
has_many :comments, dependent: :nullify
has_one :profile, dependent: :destroy

# Optional для необязательных связей
belongs_to :organization, optional: true

# Через таблицу
has_many :roles, through: :user_roles
has_many :user_roles, dependent: :destroy

# Полиморфные
has_many :comments, as: :commentable

# С условием
has_many :published_posts, -> { where(published: true) }, class_name: 'Post'
```

## Правила для Validations

```ruby
# Presence
validates :name, presence: true

# Uniqueness (всегда с индексом в БД!)
validates :email, uniqueness: { case_sensitive: false }
validates :slug, uniqueness: { scope: :organization_id }

# Format
validates :email, format: { with: URI::MailTo::EMAIL_REGEXP }
validates :phone, format: { with: /\A\+?[\d\s-]+\z/ }

# Length
validates :name, length: { minimum: 2, maximum: 100 }
validates :bio, length: { maximum: 500 }

# Numericality
validates :age, numericality: { greater_than: 0, less_than: 150 }
validates :price, numericality: { greater_than_or_equal_to: 0 }

# Inclusion
validates :status, inclusion: { in: %w[draft published archived] }

# Custom
validate :end_date_after_start_date

private

def end_date_after_start_date
  return unless start_date && end_date
  errors.add(:end_date, 'must be after start date') if end_date <= start_date
end
```

## Правила для Scopes

```ruby
# Простые
scope :active, -> { where(active: true) }
scope :recent, -> { order(created_at: :desc) }

# С параметрами
scope :created_after, ->(date) { where('created_at > ?', date) }
scope :by_status, ->(status) { where(status: status) }

# Составные
scope :active_admins, -> { active.where(role: :admin) }

# С join
scope :with_posts, -> { joins(:posts).distinct }

# Поиск (PostgreSQL)
scope :search, ->(q) {
  where('name ILIKE :q OR email ILIKE :q', q: "%#{sanitize_sql_like(q)}%")
}

# Для пагинации
scope :page, ->(num, per: 25) {
  limit(per).offset((num.to_i - 1) * per)
}
```

## Enums

```ruby
# Базовый
enum :status, { draft: 0, published: 1, archived: 2 }

# С префиксом (рекомендуется)
enum :status, { draft: 0, published: 1 }, prefix: true
# user.status_draft?, user.status_published!

# С кастомным префиксом
enum :role, { user: 0, admin: 1 }, prefix: :is
# user.is_user?, user.is_admin!

# Без scopes
enum :status, { draft: 0, published: 1 }, scopes: false
```

## Что НЕ должно быть в модели

```ruby
# ПЛОХО - бизнес-логика в модели
class User < ApplicationRecord
  def send_welcome_email  # ❌ Вынести в сервис
    UserMailer.welcome(self).deliver_later
  end

  def calculate_subscription_price  # ❌ Вынести в сервис
    # complex logic
  end

  def can_access?(resource)  # ❌ Вынести в Policy/Ability
    admin? || resource.user_id == id
  end
end

# ХОРОШО - только данные
class User < ApplicationRecord
  scope :active, -> { where(status: :active) }
  scope :premium, -> { where(subscription: :premium) }
end
```
