# Query Objects

Для сложных запросов, которые не помещаются в scope.

## Базовый класс

```ruby
# app/queries/application_query.rb
class ApplicationQuery
  DEFAULT_PER_PAGE = 25
  MAX_PER_PAGE = 100

  def self.call(...) = new(...).call

  def initialize(scope = nil, params: {})
    @scope = scope || default_scope
    @params = params
  end

  def call
    raise NotImplementedError
  end

  private

  attr_reader :scope, :params

  def default_scope
    raise NotImplementedError, "Define default_scope or pass scope"
  end

  # === Helpers ===

  def paginate
    page = [params[:page].to_i, 1].max
    per_page = [[params[:per_page].to_i, 1].max, MAX_PER_PAGE].min
    per_page = DEFAULT_PER_PAGE if per_page.zero?

    @scope = scope.limit(per_page).offset((page - 1) * per_page)
  end

  def search_in(*columns, query:)
    return if query.blank?

    conditions = columns.map { |c| "#{c} ILIKE :q" }.join(' OR ')
    @scope = scope.where(conditions, q: "%#{query}%")
  end

  def filter_by(column, value)
    return if value.blank?
    @scope = scope.where(column => value)
  end

  def filter_by_date_range(column, from:, to:)
    @scope = scope.where(column => from..to) if from && to
    @scope = scope.where("#{column} >= ?", from) if from && !to
    @scope = scope.where("#{column} <= ?", to) if !from && to
  end

  def sort_by(column, direction: :desc, allowed: [])
    column = column.to_sym
    return unless allowed.include?(column)

    direction = direction.to_s.downcase == 'asc' ? :asc : :desc
    @scope = scope.order(column => direction)
  end
end
```

## Пример Query Object

```ruby
# app/queries/users/search_query.rb
module Users
  class SearchQuery < ApplicationQuery
    SORTABLE_COLUMNS = %i[name email created_at].freeze

    def call
      filter_by_status
      filter_by_role
      filter_by_organization
      filter_by_date_range(:created_at, from: params[:from], to: params[:to])
      search
      apply_sorting
      paginate
      scope
    end

    private

    def default_scope
      User.all
    end

    def filter_by_status
      filter_by(:status, params[:status])
    end

    def filter_by_role
      filter_by(:role, params[:role])
    end

    def filter_by_organization
      filter_by(:organization_id, params[:organization_id])
    end

    def search
      search_in(:name, :email, query: params[:q])
    end

    def apply_sorting
      column = params[:sort] || :created_at
      direction = params[:direction] || :desc
      sort_by(column, direction: direction, allowed: SORTABLE_COLUMNS)
    end
  end
end
```

## Использование

```ruby
# В контроллере
def index
  users = Users::SearchQuery.call(params: filter_params)
  render_collection(users, serializer: UserSerializer)
end

def filter_params
  params.permit(:status, :role, :organization_id, :q, :from, :to, :sort, :direction, :page, :per_page)
end

# С кастомным scope
active_users = Users::SearchQuery.call(User.active, params: filter_params)

# В сервисе
def call
  users = Users::SearchQuery.call(params: search_params)
  # process users...
end
```

## Сложный Query с JOIN

```ruby
# app/queries/posts/feed_query.rb
module Posts
  class FeedQuery < ApplicationQuery
    def initialize(user:, params: {})
      @user = user
      super(nil, params: params)
    end

    def call
      filter_by_followed_authors
      filter_by_topics
      exclude_blocked
      only_published
      apply_sorting
      paginate
      scope
    end

    private

    attr_reader :user

    def default_scope
      Post.includes(:author, :topics, :comments)
    end

    def filter_by_followed_authors
      followed_ids = user.followed_user_ids
      @scope = scope.where(author_id: followed_ids)
    end

    def filter_by_topics
      return if params[:topic_ids].blank?
      @scope = scope.joins(:topics).where(topics: { id: params[:topic_ids] })
    end

    def exclude_blocked
      blocked_ids = user.blocked_user_ids
      @scope = scope.where.not(author_id: blocked_ids) if blocked_ids.any?
    end

    def only_published
      @scope = scope.where(status: :published)
    end

    def apply_sorting
      @scope = case params[:sort]
               when 'popular'
                 scope.order(comments_count: :desc, created_at: :desc)
               when 'oldest'
                 scope.order(created_at: :asc)
               else
                 scope.order(created_at: :desc)
               end
    end
  end
end
```

## Query с агрегацией

```ruby
# app/queries/analytics/user_stats_query.rb
module Analytics
  class UserStatsQuery < ApplicationQuery
    def call
      scope
        .select(
          'users.*',
          'COUNT(posts.id) as posts_count',
          'COUNT(comments.id) as comments_count',
          'MAX(posts.created_at) as last_post_at'
        )
        .left_joins(:posts, :comments)
        .group('users.id')
        .order('posts_count DESC')
    end

    private

    def default_scope
      User.active
    end
  end
end
```

## Когда использовать Query Objects

✅ **Используй Query Object когда:**
- Запрос > 3-4 условий
- Нужна пагинация + фильтрация + сортировка
- Запрос с JOIN/includes
- Запрос переиспользуется
- Нужны агрегации

❌ **НЕ нужен Query Object:**
- Простой scope достаточен
- Запрос используется один раз
- Только 1-2 условия
