# JSON:API Сериализаторы

Используем `jsonapi-serializer` для всех API ответов.

## Установка

```ruby
gem 'jsonapi-serializer'
```

```bash
rails g serializer User name email
```

## Базовый сериализатор

```ruby
# app/serializers/user_serializer.rb
class UserSerializer
  include JSONAPI::Serializer

  set_type :user

  attributes :name, :email, :created_at
end
```

## Формат ответа

```json
{
  "data": {
    "id": "1",
    "type": "user",
    "attributes": {
      "name": "John Doe",
      "email": "john@example.com",
      "created_at": "2024-01-15T10:30:00Z"
    }
  }
}
```

## Кастомные атрибуты

```ruby
class UserSerializer
  include JSONAPI::Serializer

  attributes :name, :email

  # Вычисляемый атрибут
  attribute :full_name do |user|
    "#{user.first_name} #{user.last_name}"
  end

  # Через proc
  attribute :member_since, &:created_at

  # Условный атрибут
  attribute :admin_notes, if: proc { |_user, params|
    params[:current_user]&.admin?
  }
end
```

## Отношения

```ruby
class UserSerializer
  include JSONAPI::Serializer

  attributes :name, :email

  has_many :posts
  has_many :comments
  belongs_to :organization

  # С кастомным типом
  belongs_to :manager, record_type: :user

  # Условное отношение
  has_many :drafts, if: proc { |user, params|
    params[:include_drafts]
  }
end
```

## Использование в контроллере

```ruby
# Один объект
render json: UserSerializer.new(user).serializable_hash

# Коллекция с мета-информацией
options = {
  meta: { total: users.count, page: params[:page] },
  params: { current_user: current_user }
}
render json: UserSerializer.new(users, options).serializable_hash

# С include (sideloading)
render json: UserSerializer.new(user, include: [:posts, :organization]).serializable_hash

# С пагинацией
options = {
  meta: {
    total_count: users.total_count,
    total_pages: users.total_pages,
    current_page: users.current_page
  },
  links: {
    self: api_v1_users_url(page: users.current_page),
    first: api_v1_users_url(page: 1),
    last: api_v1_users_url(page: users.total_pages),
    prev: users.prev_page ? api_v1_users_url(page: users.prev_page) : nil,
    next: users.next_page ? api_v1_users_url(page: users.next_page) : nil
  }
}
render json: UserSerializer.new(users, options).serializable_hash
```

## Кэширование

```ruby
class UserSerializer
  include JSONAPI::Serializer

  cache_options store: Rails.cache, namespace: 'jsonapi', expires_in: 1.hour

  attributes :name, :email
end
```

## Вложенные сериализаторы

```ruby
class PostSerializer
  include JSONAPI::Serializer

  attributes :title, :body, :created_at

  belongs_to :author, serializer: UserSerializer
  has_many :comments, serializer: CommentSerializer
end
```

## Хелперы в сериализаторе

```ruby
module AvatarHelper
  def avatar_url(user)
    user.avatar.attached? ? Rails.application.routes.url_helpers.url_for(user.avatar) : nil
  end
end

class UserSerializer
  include JSONAPI::Serializer
  extend AvatarHelper

  attributes :name, :email

  attribute :avatar do |user|
    avatar_url(user)
  end
end
```
