# Сервисы (Command Pattern)

Вся бизнес-логика в сервисах. Контроллеры только вызывают сервисы.

## Базовый класс

```ruby
# app/services/application_service.rb
class ApplicationService
  def self.call(...)
    new(...).call
  end

  def call
    raise NotImplementedError, "#{self.class} must implement #call"
  end

  private

  def success(data = nil)
    ServiceResult.new(success: true, data: data)
  end

  def failure(errors)
    ServiceResult.new(success: false, errors: Array(errors))
  end
end

class ServiceResult
  attr_reader :data, :errors

  def initialize(success:, data: nil, errors: [])
    @success = success
    @data = data
    @errors = errors
    freeze
  end

  def success? = @success
  def failure? = !@success

  # Pattern matching support
  def deconstruct_keys(_keys)
    { success: @success, data: @data, errors: @errors }
  end
end
```

## Пример сервиса

```ruby
# app/services/users/create_service.rb
module Users
  class CreateService < ApplicationService
    def initialize(params:, current_user: nil)
      @params = params
      @current_user = current_user
    end

    def call
      user = User.new(permitted_params)

      if user.save
        notify_admin(user)
        success(user)
      else
        failure(user.errors.full_messages)
      end
    end

    private

    attr_reader :params, :current_user

    def permitted_params
      params.slice(:email, :name, :password)
    end

    def notify_admin(user)
      AdminMailer.new_user(user).deliver_later
    end
  end
end
```

## Использование

```ruby
# В контроллере
result = Users::CreateService.call(params: user_params, current_user: current_user)

if result.success?
  render json: UserSerializer.new(result.data).serializable_hash, status: :created
else
  render json: { errors: result.errors }, status: :unprocessable_entity
end

# Pattern matching (Ruby 3+)
case Users::CreateService.call(params: params)
in ServiceResult[success: true, data:]
  render_success(data)
in ServiceResult[errors:]
  render_errors(errors)
end
```

## Правила

1. **Один публичный метод** — только `call`
2. **Возвращает ServiceResult** — success или failure
3. **Инициализация через именованные параметры** — `def initialize(params:, user:)`
4. **Приватные методы** — декомпозиция логики
5. **Без side effects в initialize** — только присвоение
6. **Транзакции при необходимости** — `ActiveRecord::Base.transaction { ... }`

## Сложный сервис с транзакцией

```ruby
module Orders
  class CreateService < ApplicationService
    def initialize(user:, items:, payment_method:)
      @user = user
      @items = items
      @payment_method = payment_method
    end

    def call
      ActiveRecord::Base.transaction do
        order = create_order
        process_payment(order)
        send_confirmation(order)
        success(order)
      end
    rescue PaymentError => e
      failure("Payment failed: #{e.message}")
    rescue => e
      failure("Order creation failed: #{e.message}")
    end

    private

    attr_reader :user, :items, :payment_method

    def create_order
      Order.create!(user: user, items: items, status: :pending)
    end

    def process_payment(order)
      PaymentGateway.charge(payment_method, order.total)
      order.update!(status: :paid)
    end

    def send_confirmation(order)
      OrderMailer.confirmation(order).deliver_later
    end
  end
end
```
