# RSpec Тестирование

Full test coverage с RSpec, FactoryBot, shoulda-matchers.

## Установка

```ruby
# Gemfile
group :development, :test do
  gem 'rspec-rails'
  gem 'factory_bot_rails'
  gem 'faker'
end

group :test do
  gem 'shoulda-matchers'
  gem 'database_cleaner-active_record'
end
```

```bash
rails g rspec:install
```

## Конфигурация

```ruby
# spec/rails_helper.rb
RSpec.configure do |config|
  config.include FactoryBot::Syntax::Methods
end

Shoulda::Matchers.configure do |config|
  config.integrate do |with|
    with.test_framework :rspec
    with.library :rails
  end
end
```

## FactoryBot Фабрики

```ruby
# spec/factories/users.rb
FactoryBot.define do
  factory :user do
    sequence(:email) { |n| "user#{n}@example.com" }
    name { Faker::Name.name }
    password { 'password123' }
    status { :active }
    role { :user }

    # Traits
    trait :admin do
      role { :admin }
    end

    trait :pending do
      status { :pending }
    end

    trait :with_organization do
      association :organization
    end

    trait :with_posts do
      transient do
        posts_count { 3 }
      end

      after(:create) do |user, ctx|
        create_list(:post, ctx.posts_count, author: user)
      end
    end
  end
end

# Использование
user = create(:user)
admin = create(:user, :admin)
user_with_posts = create(:user, :with_posts, posts_count: 5)
users = create_list(:user, 3)
attrs = attributes_for(:user)
```

## Тесты моделей

```ruby
# spec/models/user_spec.rb
RSpec.describe User, type: :model do
  describe 'associations' do
    it { is_expected.to belong_to(:organization).optional }
    it { is_expected.to have_many(:posts).dependent(:destroy) }
    it { is_expected.to have_many(:comments).dependent(:destroy) }
  end

  describe 'validations' do
    subject { build(:user) }

    it { is_expected.to validate_presence_of(:email) }
    it { is_expected.to validate_uniqueness_of(:email).case_insensitive }
    it { is_expected.to validate_presence_of(:name) }
    it { is_expected.to validate_length_of(:name).is_at_least(2).is_at_most(100) }
  end

  describe 'enums' do
    it { is_expected.to define_enum_for(:status).with_values(pending: 0, active: 1, suspended: 2) }
    it { is_expected.to define_enum_for(:role).with_values(user: 0, admin: 1) }
  end

  describe 'scopes' do
    describe '.active' do
      it 'returns only active users' do
        active = create(:user, status: :active)
        pending = create(:user, :pending)

        expect(described_class.active).to include(active)
        expect(described_class.active).not_to include(pending)
      end
    end

    describe '.search' do
      it 'finds users by name or email' do
        john = create(:user, name: 'John Doe')
        jane = create(:user, email: 'jane@example.com')
        bob = create(:user, name: 'Bob')

        expect(described_class.search('john')).to include(john)
        expect(described_class.search('jane')).to include(jane)
        expect(described_class.search('john')).not_to include(bob)
      end
    end
  end

  describe 'callbacks' do
    it 'normalizes email before save' do
      user = create(:user, email: '  TEST@Example.COM  ')
      expect(user.email).to eq('test@example.com')
    end
  end
end
```

## Тесты сервисов

```ruby
# spec/services/users/create_service_spec.rb
RSpec.describe Users::CreateService, type: :service do
  describe '.call' do
    subject(:result) { described_class.call(params: params) }

    context 'with valid params' do
      let(:params) { { email: 'new@example.com', name: 'New User', password: 'password123' } }

      it 'returns success' do
        expect(result).to be_success
      end

      it 'creates a user' do
        expect { result }.to change(User, :count).by(1)
      end

      it 'returns the created user' do
        expect(result.data).to be_a(User)
        expect(result.data.email).to eq('new@example.com')
      end
    end

    context 'with invalid params' do
      let(:params) { { email: '', name: '' } }

      it 'returns failure' do
        expect(result).to be_failure
      end

      it 'does not create a user' do
        expect { result }.not_to change(User, :count)
      end

      it 'returns errors' do
        expect(result.errors).to include(/email/i)
      end
    end

    context 'with duplicate email' do
      let!(:existing) { create(:user, email: 'taken@example.com') }
      let(:params) { { email: 'taken@example.com', name: 'New User' } }

      it 'returns failure with email error' do
        expect(result).to be_failure
        expect(result.errors.join).to match(/email/i)
      end
    end
  end
end
```

## Request Specs

```ruby
# spec/requests/api/v1/users_spec.rb
RSpec.describe 'Api::V1::Users', type: :request do
  let(:json) { JSON.parse(response.body, symbolize_names: true) }

  describe 'GET /api/v1/users' do
    let!(:users) { create_list(:user, 3) }

    it 'returns list of users' do
      get '/api/v1/users'

      expect(response).to have_http_status(:ok)
      expect(json[:data].size).to eq(3)
    end

    it 'filters by status' do
      active = create(:user, status: :active)
      pending = create(:user, :pending)

      get '/api/v1/users', params: { status: 'active' }

      ids = json[:data].pluck(:id)
      expect(ids).to include(active.id.to_s)
      expect(ids).not_to include(pending.id.to_s)
    end
  end

  describe 'GET /api/v1/users/:id' do
    let(:user) { create(:user) }

    it 'returns the user' do
      get "/api/v1/users/#{user.id}"

      expect(response).to have_http_status(:ok)
      expect(json[:data][:id]).to eq(user.id.to_s)
      expect(json[:data][:attributes][:email]).to eq(user.email)
    end

    it 'returns 404 for non-existent user' do
      get '/api/v1/users/99999'

      expect(response).to have_http_status(:not_found)
    end
  end

  describe 'POST /api/v1/users' do
    let(:valid_params) do
      { user: { email: 'new@example.com', name: 'New User', password: 'password123' } }
    end

    context 'with valid params' do
      it 'creates a user' do
        expect {
          post '/api/v1/users', params: valid_params
        }.to change(User, :count).by(1)

        expect(response).to have_http_status(:created)
        expect(json[:data][:attributes][:email]).to eq('new@example.com')
      end
    end

    context 'with invalid params' do
      let(:invalid_params) { { user: { email: '' } } }

      it 'returns errors' do
        post '/api/v1/users', params: invalid_params

        expect(response).to have_http_status(:unprocessable_entity)
        expect(json[:errors]).to be_present
      end
    end
  end

  describe 'PATCH /api/v1/users/:id' do
    let(:user) { create(:user, name: 'Old Name') }

    it 'updates the user' do
      patch "/api/v1/users/#{user.id}", params: { user: { name: 'New Name' } }

      expect(response).to have_http_status(:ok)
      expect(user.reload.name).to eq('New Name')
    end
  end

  describe 'DELETE /api/v1/users/:id' do
    let!(:user) { create(:user) }

    it 'deletes the user' do
      expect {
        delete "/api/v1/users/#{user.id}"
      }.to change(User, :count).by(-1)

      expect(response).to have_http_status(:no_content)
    end
  end
end
```

## Shared Examples

```ruby
# spec/support/shared_examples/service_examples.rb
RSpec.shared_examples 'a successful service' do
  it 'returns success' do
    expect(result).to be_success
    expect(result.errors).to be_empty
  end
end

RSpec.shared_examples 'a failing service' do |expected_error|
  it 'returns failure' do
    expect(result).to be_failure
    expect(result.errors).not_to be_empty
  end

  if expected_error
    it "includes error about #{expected_error}" do
      expect(result.errors.join).to match(/#{expected_error}/i)
    end
  end
end

# Использование
RSpec.describe Users::CreateService do
  describe '.call' do
    subject(:result) { described_class.call(params: params) }

    context 'with valid params' do
      let(:params) { attributes_for(:user) }
      it_behaves_like 'a successful service'
    end

    context 'with blank email' do
      let(:params) { { email: '', name: 'Test' } }
      it_behaves_like 'a failing service', 'email'
    end
  end
end
```

## Хелперы

```ruby
# spec/support/request_helpers.rb
module RequestHelpers
  def json_response
    JSON.parse(response.body, symbolize_names: true)
  end

  def json_data
    json_response[:data]
  end

  def json_errors
    json_response[:errors]
  end

  def auth_headers(user)
    token = Users::GenerateTokenService.call(user: user).data
    { 'Authorization' => "Bearer #{token}" }
  end
end

RSpec.configure do |config|
  config.include RequestHelpers, type: :request
end
```
